"""
    Created Mar 2014
    This file is part of VisualCNA
    Copyright (2014) Prakash Chandra Rathi and Daniel Mulnaes

    VisualCNA is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    VisualCNA is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
#####################################################################
# -*- coding: utf-8 -*-
#
# Authors: Prakash Chandra Rathi and Daniel Mulnaes
# Heinrich Heine University, Duesseldorf
# Institute for Pharmaceutical and Medicinal Chemistry
# Universitaetsstr. 1 40225 Duesseldorf
# Germany
#####################################################################


# Standard python libraries
import os, glob, time, subprocess, signal, string, random
from threading import Thread
from Tkinter import *
from tkFileDialog import asksaveasfile

# External Modules
import numpy as np
from pymol import cmd
from pymol.wizard import Wizard
from tkintertable.TableModels import TableModel

# VisualCNA Modules
import cna_settings, read_pdb, table, errors, show_network

class state_label(Wizard):

    def __init__(self, hb_cut_offs, _self=cmd):
        Wizard.__init__(self, _self)
        self.hb_cut_offs=hb_cut_offs
    def get_prompt(self):
        new_state=cmd.get_state()
        msg_state='E cut_off = %.2f'%self.hb_cut_offs[new_state-1]
        self.prompt=[msg_state]
        return self.prompt

# Level 1 Interface functions that are only called by visualCNA

def search(parent, direction):
    parent.table_canvas.findValue(parent.table_search.get(), direction)

def write_pymol_net(parent):
    save_dir_file=asksaveasfile(title='Save Network As',
                                             initialdir=parent.data_dir,
                                             filetypes=[("Network files", "*network.dat")],
                                             defaultextension='_network.dat')

    parent.net_obj.write(save_dir_file.name)
    print 'Saved network as %s'%save_dir_file.name

def validate_type(parent, args, type_=None):
    # Parse argument string
    try:
        args=dict([i.split(':') for i in args.split(',')])
        valid=True
    except:
        args=args.replace(',,', ',').replace(':,', ',').replace('::', ':').split(',')
        args=dict([i.split(':') for i in args[:4]+[args[4].replace(':', '')+':']+args[5:]])
        valid=False

    if parent.parent.nametowidget(args['W']).config()['state'][-1]=='disabled':
        return True

    # Test if parsing was valid and if input conforms to type
    if valid and type_=='numerical':
        try:
            args['P'] in '{}-' or float(args['P'])
        except:
            valid=False
    elif valid and type_=='integer':
        try:
            args['P'] in '{}-' or int(args['P'])
        except:
            valid=False

    # Raise error if input is not valid
    if not valid:
        parent.parent.bell()
        errors.input_error(error_id=0, args=(type_,))

    if args['V']=='focusout': valid=False

    parent.parent.after_idle(parent.parent.nametowidget(args['W']).config, {'validate':args['v']})
    return valid

def validate_range(parent, args, range_, default):
    # Parse argument string and get field and value
    args=dict([i.split(':') for i in args.split(',')])
    field=parent.parent.nametowidget(args['W'])
    value=field.get()

    # Test if value conforms to range and give error if not
    try:
        if field.config()['state'][:-1]!='disabled' and range_ is not None and not range_[0]<=float(value)<=range_[1]:
            parent.parent.bell()
            set_field(field, default)
            field.config(bg='white')
            errors.input_error(error_id=1, args=range_)
        elif str(value)!=str(default):
            field.config(bg=field.config()['highlightcolor'][-1])
        else:
            field.config(bg='white')

    except:
        parent.parent.bell()
        errors.input_error(error_id=0, args=('numerical',))
        set_field(field, default)

    parent.parent.after_idle(parent.parent.nametowidget(args['W']).config, {'validate':args['v']})

def validate_constraint(parent, args, range_, default):
    try:
        args=dict([i.split(':') for i in args.split(',')])
    except:
        args=args.replace(',,', ',').replace(':,', ',').replace('::', ':').split(',')
        args=args=dict([i.split(':') for i in args[:4]+[args[4].replace(':', '')+':']+args[5:]])

    field=parent.parent.nametowidget(args['W'])
    value=args['P']
    valid=False

    # Do Entry Validation
    if not hasattr(parent, 'net_obj'):
        return valid

    valid=value in ['{}', ''] or (value.isdigit() and range_[0]<=int(value)<=range_[1])

    if not valid:
        parent.parent.bell()
        errors.input_error(error_id=2, args=range_)
        set_field(field, default)

    # Do Auto-fill of label and energy field if pattern is found
    if valid:
        try:
            atom_id_1=int(value) if field==parent.add_atom_1 else int(parent.add_atom_1.get())
            atom_id_2=int(value) if field==parent.add_atom_2 else int(parent.add_atom_2.get())
            if range_[0]<=atom_id_1<=range_[1] and range_[0]<=atom_id_2<=range_[1]:
                ctype=parent.net_obj.short_to_long[parent.table_constraints.get()[:2]]

                if ctype in ['HBOND', 'SBRIDGE', 'HPHOBES']:
                    label1=parent.net_obj.get_label(ctype, atom_id_1, atom_id_2)
                    label2=parent.net_obj.get_label(ctype, atom_id_2, atom_id_1)

                    if label1 in parent.net_obj.label_to_id:
                        constraint=parent.net_obj.constraints[parent.net_obj.label_to_id[label1]]
                    elif label2 in parent.net_obj.label_to_id:
                        constraint=parent.net_obj.constraints[parent.net_obj.label_to_id[label2]]
                    else:
                        constraint=None
                elif ctype in ['CUSTOM']:
                    label=parent.net_obj.get_label(ctype, atom_id_1, atom_id_2)
                    bars=int(parent.add_c_bars.get()[0])
                    constraint={'bars':bars, 'energy':None, 'label':label}

                if constraint is not None:
                    energy=str(round(float(constraint['energy']), 2)) if constraint['energy'] is not None else ''
                    bars=str(constraint['bars'])
                    set_field(parent.add_clabel, constraint['label'])
                    set_field(parent.add_energy, energy)
                    parent.add_c_bars.set(bars)
                else:
                    set_field(parent.add_clabel, '')
                    set_field(parent.add_energy, '')
        except:
            set_field(parent.add_clabel, '')
            set_field(parent.add_energy, '')

    if args['V']=='focusout': valid=False

    parent.parent.after_idle(parent.parent.nametowidget(args['W']).config, {'validate':args['v']})

    return valid

def set_cna_type(parent, menu):

    def hide_fnc(hide):
        if hide:
            parent.fnc_num_b.config(text='', state='disabled', relief='flat')
            parent.cna_args['fnc_num'].config(relief='flat', highlightthickness=0, state='disabled')
            set_field(parent.cna_args['fnc_num'], '')
        else:
            parent.fnc_num_b.config(text='FNC Networks', state='normal', relief='raised')
            parent.cna_args['fnc_num'].config(relief='sunken', highlightthickness=1, state='normal')
            set_field(parent.cna_args['fnc_num'], str(parent.analyze_settings.fnc_default))

    def set_pdb_input(mode):
        if mode=='file':
            background_color='black' if os.path.isfile(parent.open_pdb.get()) else 'grey'
            parent.open_pdb_b.field_mode='file'
            parent.open_pdb_drop_m.config(state='normal')
            if parent.open_pdb_drop.get()=='':
                parent.open_pdb_drop.set('File')
                parent.open_pdb_b.config(state='normal', text='PDB File')
                parent.open_pdb.config(disabledforeground=background_color)
            elif parent.open_pdb_drop.get()=='File':
                parent.open_pdb.config(disabledforeground=background_color)
                parent.open_pdb_b.config(state='normal', text='PDB File')
            else:
                parent.open_pdb_b.config(state='disabled', text='PDB File')
                parent.open_pdb.config(disabledforeground=background_color)
        elif mode=='dir':
            background_color='black' if os.path.isdir(parent.open_pdb.get()) else 'grey'
            parent.open_pdb_b.field_mode='dir'
            parent.open_pdb_drop_m.config(state='disabled')
            parent.open_pdb_drop.set('')
            parent.open_pdb_b.config(state='normal', text='Ensemble')
            if background_color=='grey':
                set_field(parent.open_pdb, parent.data_dir)
            parent.open_pdb.config(disabledforeground=background_color)

    if menu==parent.analyze_settings.cna_types[0]:  # For SNT Runs
        parent.all_results_b.config(state='disabled')
        parent.out_network_b.config(text='Write FIRST Network for single PDB-structure')
        hide_fnc(True)
        set_pdb_input('file')
        set_net_input(parent, parent.open_net_drop.get(), 'open_net')

    elif menu==parent.analyze_settings.cna_types[1]:  # For ENT Runs
        parent.all_results_b.config(state='normal')
        parent.out_network_b.config(text='Write FIRST Network for each ensemble structure')
        hide_fnc(True)
        set_pdb_input('dir')
        set_net_input(parent, parent.open_net_drop.get(), 'open_net')

    elif menu==parent.analyze_settings.cna_types[2]:  # For FNC Runs
        parent.all_results_b.config(state='normal')
        parent.out_network_b.config(text='Write FIRST Network for each fuzzy network')
        hide_fnc(False)
        set_pdb_input('file')
        set_net_input(parent, parent.open_net_drop.get(), 'open_net')
    parent.cna_args['cna_type'].set(menu)

def set_pdb_input(parent, i, field, type_specific=True):
    if i=='File':
        if 'show' not in field:
            getattr(parent, field.replace('pdb', 'net')+'_drop').set('None')
        disable=parent.cna_args['cna_type'].get()==parent.analyze_settings.cna_types[1] and type_specific
        getattr(parent, field+'_b').config(state='disabled' if disable else 'normal')
        getattr(parent, field).config(disabledforeground='black' if os.path.isfile(parent.open_pdb.get()) else 'grey')
    else:
        if 'show' not in field and hasattr(parent, 'net_obj'):
            getattr(parent, field.replace('pdb', 'net')+'_drop').set('PyMOL')
            getattr(parent, field.replace('pdb', 'net')+'_b').config(state='disabled')
        getattr(parent, field).config(disabledforeground='grey')
        getattr(parent, field+'_b').config(state='disabled')
    if getattr(parent, field.replace('pdb', 'net')+'_drop').get() in ['None', 'File']:
        getattr(parent, field.replace('pdb', 'net')+'_b').config(state='disabled')

def set_net_input(parent, i, field):
    background_color='black' if os.path.isfile(getattr(parent, field).get()) else 'grey'
    getattr(parent, field+'_b').config(state='disabled' if i in ['None', 'PyMOL', 'PDB'] else 'normal')
    getattr(parent, field).config(disabledforeground=background_color)

def set_hp(parent, menu):

    def hide_hydro(field, hide, default):
        if hide:
            getattr(parent, field+'_b').config(text='', state='disabled', relief='flat')
            parent.cna_args[field].config(relief='flat', highlightthickness=0, state='disabled')
            set_field(parent.cna_args[field], '')
        else:
            getattr(parent, field+'_b').config(text=default, state='normal', relief='raised')
            parent.cna_args[field].config(relief='sunken', highlightthickness=1, state='normal')
            if field=='hp_start':
                set_field(parent.cna_args[field], str(parent.analyze_settings.hp_cut_defaults[0]))
            else:
                set_field(parent.cna_args[field], str(parent.analyze_settings.hp_cut_defaults[1]))

    if menu in parent.analyze_settings.tus_types+parent.analyze_settings.tet_types[1:]:
        if parent.cna_args['tus_type'].get()==parent.analyze_settings.tus_types[0]:
            hide_hydro('hp_start', False, 'Constant')
            hide_hydro('hp_stop', True, 'Terminal')
        elif parent.cna_args['tus_type'].get()==parent.analyze_settings.tus_types[1]:
            hide_hydro('hp_start', False, 'Initial')
            hide_hydro('hp_stop', False, 'Terminal')

    elif menu==parent.analyze_settings.tet_types[0]:
        hide_hydro('hp_start', True, 'Initial')
        hide_hydro('hp_stop', True, 'Terminal')

def constraint_undo(parent):
    try:    parent.net_obj.undo()
    except: pass

def constraint_redo(parent):
    try:    parent.net_obj.redo()
    except: pass

# Modify input for structure input selection drop-down depending on whether PyMOL structures are loaded
def detect_structure_update(parent, var, menu, defaults=[], command=None, prefix=''):
    pymol_loaded=[i for i in cmd.get_names_of_type('object:molecule') if 'RC' not in i]

    def pass_command(label):
        getattr(parent, var).set(prefix+label)
        if command is not None:
            command(label)

    getattr(parent, menu)['menu'].delete(0, 'end')
    for s in pymol_loaded:
        getattr(parent, menu)['menu'].add_command(
                    label=s,
                    command=lambda x=s:pass_command(x))
    for d in defaults:
        getattr(parent, menu)['menu'].add_command(
                    label=d,
                    command=lambda x=d:pass_command(x))

# Modify input for network input selection drop-down depending on whether PyMOL structures are selected
def detect_structure_exists(parent, var, menu, defaults=[], command=None):

    def pass_command(label):
        getattr(parent, var).set(label)
        if command is not None:
            command(label)

    getattr(parent, menu)['menu'].delete(0, 'end')
    for d in defaults:
        getattr(parent, menu)['menu'].add_command(
                    label=d,
                    command=lambda x=d:pass_command(x))

    if parent.open_pdb_drop.get()!='File' and hasattr(parent, 'net_obj'):
        getattr(parent, menu)['menu'].add_command(
                    label='PyMOL',
                    command=lambda x='PyMOL':pass_command(x))

def detect_make_constraint_update(parent, bond_type):
    ctype=bond_type.split('-')[0].strip()
    cbars=cna_settings.analyze_setup.barnum_defaults[ctype]
    cbars=str(cbars)+' Bar' if cbars==1 else str(cbars)+' Bars'
    parent.bar_select.set(cbars)
    if ctype=='CV':
        cbars=[str(i)+' Bar' if i==1 else str(i)+' Bars' for i in range(1, 7)]
    else:
        cbars=[cbars]

    parent.bar_select_menu['menu'].delete(0, 'end')
    for bar in cbars:
        parent.bar_select_menu['menu'].add_command(
                label=bar,
                command=lambda x=bar:parent.bar_select.set(x))

def make_table_constraint(parent):
    try:
        label=parent.add_clabel.get()
        (atom_id_1, atom_id_2)=sorted([int(parent.add_atom_1.get()), int(parent.add_atom_2.get())])
        if (atom_id_1, atom_id_2) in parent.net_obj.constraints:
            bars=parent.net_obj.constraints[(atom_id_1, atom_id_2)]['bars']
            label=parent.net_obj.constraints[(atom_id_1, atom_id_2)]['label']
        else:
            bars=int(parent.add_c_bars.get()[0])

        parent.net_obj.add_constraint(label, bars, atom_id_1, atom_id_2)
    except:
        pass

def make_pymol_constraint(parent):
    if 'sele' in cmd.get_names('all'):
        ids=list(set(cmd.identify('sele')))
        if len(ids)==2:
            ctype=parent.net_obj.short_to_long[parent.make_constraint.get()[:2]]
            if ctype in ['HBOND', 'SBRIDGE']:
                if 'H' in parent.pdb_obj.atom_details[ids[0]][1]:
                    atom_id_1=ids[0]
                    atom_id_2=ids[1]
                else:
                    atom_id_1=ids[1]
                    atom_id_2=ids[0]
            else:
                atom_id_1=min(ids)
                atom_id_2=max(ids)

            label=parent.net_obj.get_label(ctype, atom_id_1, atom_id_2)
            bars=int(parent.bar_select.get().split()[0])
            parent.net_obj.add_constraint(label, bars, atom_id_1, atom_id_2)
        else:
            errors.attention(9)

def delete_pymol_constraint(parent):
    if 'sele' in cmd.get_names('all'):
        ids=list(set(cmd.identify('sele')))
        if len(ids)==2:
            try:
                label=parent.net_obj.constraints[(ids[0], ids[1])]['label']
            except:
                try:
                    label=parent.net_obj.constraints[(ids[1], ids[0])]['label']
                except:
                    print 'No constraint was found between atoms %d and %d.'%(min(ids), max(ids))
                    return

            parent.net_obj.delete_constraints([label], log_update=True, table_update=True)
        else:
            errors.attention(9)

def compare_data(parent):
    if parent.cna_exe is None:
        errors.attention(11)
        return

    def check_path(field, save=False):
        field_directory=field.get()
        if os.path.isdir(field_directory+'results/'):
            set_field(field, field_directory+'results/')
            field_directory=field_directory+'results/'
        results_found=len(glob.glob(field_directory+'*local_indices.dat'))>0

        if not os.path.isdir(field_directory):
            print 'Error, comparison directory does not exist'
            return None, None, None
        elif results_found:
            print 'Found local index in %s, assuming data has been generated'%field_directory
        elif save:
                cmd.set('retain_order', False)
                cmd.set('pdb_retain_ids', False)
                cmd.save(field_directory+parent.pdb_obj.pdb_id+'.pdb', parent.pdb_obj.pdb_id)
                parent.net_obj.write('%s/%s_network.dat'%(field_directory, parent.pdb_obj.pdb_id))
                cmd.set('retain_order', True)
                cmd.set('pdb_retain_ids', True)
        try:
            residue_number=len(read_pdb.ReadPDB(glob.glob(field_directory+'*.pdb')[0]).res_objects)
        except:
            residue_number=None
        return field_directory, results_found, residue_number

    ref_path, ref_found, ref_length=check_path(parent.open_reference_dir)
    com_path, com_found, com_length=check_path(parent.open_comparison_dir, True)

    if ref_path==com_path:
        print 'Error, same reference and comparison directory'
        return
    elif ref_length is None:
        print 'error reference pdb not found'
        return
    elif com_length is None:
        print 'error reference pdb not found'
        return
    elif ref_length!=com_length:
        errors.input_error(28)
        return
    else:
        if not ref_found and not com_found and errors.attention(6, "Both comparison and reference results"):
            auto_run(parent, parent.compare_type.get().lower(), parent.compare, ref_path, 'Compare Data Using')
            auto_run(parent, parent.compare_type.get().lower(), parent.compare, com_path, 'Compare Data Using')
        elif not ref_found and errors.attention(6, "Reference results"):
            auto_run(parent, parent.compare_type.get().lower(), parent.compare, ref_path, 'Compare Data Using')
        elif not com_found and errors.attention(6, "Comparison results"):
            auto_run(parent, parent.compare_type.get().lower(), parent.compare, com_path, 'Compare Data Using')

        compare_observer(parent, parent.open_reference_dir, parent.open_comparison_dir, ref_found, com_found)

def auto_run(parent, run_type, button, data_dir, default_label):
    try:
        pdb_file=glob.glob(data_dir+'*.pdb')[0]
        net_file=glob.glob(data_dir+'*_network.dat')[0]
    except:
        print 'Missing pdb or network file in: '+data_dir
        return

    # Save cna run settings
    old_pdb_source=parent.open_pdb_drop.get()
    old_net_source=parent.open_net_drop.get()
    old_pdb_infile=parent.open_pdb.get()
    old_net_infile=parent.open_net.get()
    old_cna_source=parent.cna_args['cna_type'].get()
    old_out_resdir=parent.cna_args['res_dir'].get()

    # Set new settings for standardized runs
    parent.open_pdb_drop.set('File')
    parent.open_net_drop.set('File')
    set_pdb_input(parent, 'File', 'open_pdb')
    set_net_input(parent, 'File', 'open_net')
    set_field(parent.open_pdb, pdb_file)
    set_field(parent.open_net, net_file)
    set_field(parent.cna_args['res_dir'], data_dir)

    # Set CNA run type
    if run_type=='snt' and parent.cna_args['cna_type'].get()!='SNT Analysis':
        parent.cna_args['cna_type'].set('SNT Analysis')
        set_cna_type(parent, 'SNT Analysis')
    if run_type=='fnc' and parent.cna_args['cna_type'].get()!='FNC Analysis':
        parent.cna_args['cna_type'].set('FNC Analysis')
        set_cna_type(parent, 'FNC Analysis')

    run_cna(parent, button, multi=True, default_label=default_label)

    # Set back settings to the original
    parent.open_pdb_drop.set(old_pdb_source)
    parent.open_net_drop.set(old_net_source)
    set_field(parent.open_pdb, old_pdb_infile)
    set_field(parent.open_net, old_net_infile)
    parent.cna_args['cna_type'].set(old_cna_source)
    set_field(parent.cna_args['res_dir'], old_out_resdir)

# Calls set_field, run_cna, compare_observer from the outside

# Small methods functions

def get_hb_cut_offs(hb_cuts):
    (e_start, e_stop, e_step)=hb_cuts
    return [round(i, 2) for i in list(np.arange(e_start, e_stop-0.01, e_step*-1))]

def get_hp_cut_offs(hp_cuts, hb_cut_offs, tus_type):
    (c_start, c_stop)=hp_cuts
    if tus_type==1:
        return [c_start]
    elif tus_type==2:
        # 6.0: number of hydrophobic contacts are increased linearly from c_start at E = 0.0 to c_stop at E = -6.0
        hp_cut_offs=[]
        for E_cut in hb_cut_offs:
            hp_cut=float(c_start-(E_cut*(c_stop-c_start)/6.0))
            if hp_cut<c_start:
                hp_cut=c_start
            if hp_cut>c_stop:
                hp_cut=c_stop
            hp_cut_offs.append(hp_cut)
        return hp_cut_offs

# may be use biopython seq3/seq1
def get_single_resi_codes(resname):
    try:
        return {'ALA':'A', 'CYS':'C', 'ASP':'D', 'GLU':'E', 'PHE':'F', 'GLY':'G',
                'HIS':'H', 'ILE':'I', 'LYS':'K', 'LEU':'L', 'MET':'M', 'ASN':'N',
                'PRO':'P', 'GLN':'Q', 'ARG':'R', 'SER':'S', 'THR':'T', 'VAL':'V',
                'TRP':'W', 'TYR':'Y', 'HIE':'H', 'HID':'H', 'HIP':'H', 'CYX':'C'}[resname]
    except:
        return 'X'

def get_three_letter_resi_codes(resname):
    try:
        return {'A':'ALA', 'C':'CYS', 'D':'ASP', 'E':'GLU', 'F':'PHE', 'G':'GLY',
                'H':'HIS', 'I':'ILE', 'K':'LYS', 'L':'LEU', 'M':'MET', 'N':'ASN',
                'P':'PRO', 'Q':'GLN', 'R':'ARG', 'S':'SER', 'T':'THR', 'V':'VAL',
                'W':'TRP', 'Y':'TYR'}[resname]
    except:
        return 'XXX'

def set_field(field, string, disabledforeground=False):
    state=field.config()['state'][-1]
    field.config(state='normal')
    field.delete(0, len(field.get())+1)
    field.insert(0, string)
    field.config(state=state)
    if disabledforeground:
        field.config(disabledforeground="grey")
    else:
        field.config(disabledforeground="black")
    field.xview_moveto(1.0)

def get_random_string(str_length=20):
    char_set=string.ascii_uppercase+string.ascii_lowercase+string.digits
    return ''.join(random.sample(char_set*6, str_length))

def load_pdb(pdb_file, pdbid):
    cmd.load(pdb_file, pdbid, 0)
    cmd.show('cartoon', pdbid)
    cmd.color('grey50', pdbid)

# Big method functions

def detect_data_dir_update(parent, data_dir):
    # try to guess cna type
    parameter_file=data_dir+'/parameters.out'
    try:
        if "--snt" in file(parameter_file).read() or "--dilution" in file(parameter_file).read():
            cna_type="single"
        else:
            cna_type="ensemble"
    except:
        cna_type=parent.cna_type

    pdb_found=glob.glob(data_dir+'*.pdb')
    net_found=glob.glob(data_dir+'*_network.dat')
    if cna_type=="single":
        stb_found=glob.glob(data_dir+'*_stability_map.dat')
        unf_found=glob.glob(data_dir+'*_unfolding_nuclei.dat')
        li_found=glob.glob(data_dir+'*_local_indices.dat')
        gi_found=glob.glob(data_dir+'*_global_indices.dat')
    else:
        stb_found=glob.glob(data_dir+'*ent_stability_map.dat')+glob.glob(data_dir+'*fnc_stability_map.dat')+glob.glob(data_dir+'*fcd_stability_map.dat')
        unf_found=glob.glob(data_dir+'*ent_unfolding_nuclei.dat')+glob.glob(data_dir+'*fnc_unfolding_nuclei.dat')+glob.glob(data_dir+'*fcd_unfolding_nuclei.dat')
        li_found=glob.glob(data_dir+'*ent_local_indices.dat')+glob.glob(data_dir+'*fnc_local_indices.dat')+glob.glob(data_dir+'*fcd_local_indices.dat')
        gi_found=glob.glob(data_dir+'*ent_global_indices.dat')+glob.glob(data_dir+'*fnc_global_indices.dat')+glob.glob(data_dir+'*fnc_global_indices.dat')

    if net_found!=[]:
        parent.show_net_drop.set('File')
        set_net_input(parent, 'File', 'show_net')
    else:
        parent.show_net_drop.set('PDB')
        set_net_input(parent, 'PDB', 'show_net')

    set_field(parent.show_pdb, pdb_found[0] if pdb_found!=[] else data_dir, pdb_found==[])
    set_field(parent.show_net, net_found[0] if net_found!=[] else data_dir, net_found==[])
    set_field(parent.li_plot.open_file, li_found[0] if li_found!=[] else data_dir, li_found==[])
    set_field(parent.gi_plot.open_file, gi_found[0] if gi_found!=[] else data_dir, gi_found==[])
    set_field(parent.stb_plot.open_file, stb_found[0] if stb_found!=[] else data_dir, stb_found==[])
    set_field(parent.unf_plot.open_file, unf_found[0] if unf_found!=[] else data_dir, unf_found==[])

def detect_table_update(parent):
    # recreate table otherwise rows are appended to the previous rows
    parent.table_canvas=load_table(parent, parent.table_frame)
    if not hasattr(parent, 'net_obj'):
        return
    ctype=parent.net_obj.short_to_long[parent.table_constraints.get().split('-')[0].strip()]
    table_data=parent.net_obj.get_table(ctype)
    parent.table_canvas.net_parent=parent
    parent.table_canvas.model.importDict(table_data, namefield='Name')
    parent.table_canvas.redrawTable()
    parent.table_canvas.sortTable(sortcol=1)
    try:
        cmd.enable("sele")
    except:
        pass

def load_table(parent, canvas):
    table_settings={'columnnames':{1:'1', 2:'2', 3:'3', 4:'4', 5:'5'},
                    'columnlabels':{1:'Name', 2:'Atom 1', 3:'Atom 2', 4:'Bars', 5:'Energy'},
                    'columnorder':{1:'Name', 2:'Atom 1', 3:'Atom 2', 4:'Bars', 5:'Energy'},
                    'columntypes':{'Name':'text', 'Atom 1':'number', 'Atom 2':'number', 'Bars':'number', 'Energy':'number'}
                     }

    table_model=TableModel(newdict=table_settings)
    # add columwidths to the table_model
    table_model.columnwidths={'Name': 220, 'Atom 1': 60, 'Atom 2': 60, 'Bars': 60, 'Energy':60}

    table_canvas=table.MyTableCanvas(canvas, table_model, namefield='Name',
                    cellwidth=100, thefont=('Arial', 12), rowheight=18, rowsperpage=100, editable=False, paging=1,
                    align='e', width=canvas.config()['width'][-1], height=canvas.config()['height'][-1])
    table_canvas.tablecolheader=table.MyColumnHeader(canvas, table_canvas)
    table_canvas.tablerowheader=table.MyRowHeader(canvas, table_canvas)
    table_canvas.tablerowheader.checkbox_mapping={}
    table_canvas.tablerowheader.net_parent=parent
    table_canvas.createTableFrame()
    return table_canvas

def get_pymol_structure(parent, dropdown, default):
    selected_save=getattr(parent, dropdown).get().split(':')[-1].strip()
    if selected_save=='':
        getattr(parent, dropdown).set(default)
        errors.input_error(4)
        return None
    elif selected_save=='File':
        return None
    elif selected_save not in [i for i in cmd.get_names_of_type('object:molecule') if 'RC' not in i]:
        getattr(parent, dropdown).set(default)
        errors.input_error(5)
        return None
    else:
        return selected_save

def set_cna_parameters(parent, filename):
    parameters={}
    f=open(filename, 'r')
    for line in f.readlines():
        if not line.startswith('#'):
            line=line.rstrip().replace('[', '').replace(']', '').replace(',', '').replace("'", '')
            try:
                parameters[line.split()[0]]=line.split()[1:]
            except:
                pass
    f.close()
    c_values=[float(i) for i in parameters['c_values']]

    if len(c_values)==1:
        c_values.append(parent.analyze_settings.hp_cut_defaults[1])
    e_values=[float(i) for i in parameters['e_values']]
    if len(e_values)==2:
        e_values.append(parent.analyze_settings.hb_cut_defaults[2])
    elif e_values[2]<=parent.analyze_settings.hb_step_range[0]:
        e_values[2]=parent.analyze_settings.hb_step_range[0]
    if parameters['dilution'][0]=='True':
        parent.cna_type='single'
    else:
        parent.cna_type='ensemble'

    parent.hb_cut_offs=get_hb_cut_offs((e_values[1], e_values[0], e_values[2]))
    parent.hp_cut_offs=get_hp_cut_offs((c_values[0], c_values[1]), parent.hb_cut_offs, int(parameters['tus_type'][0]))

def display_network(parent, stb_file, gi_file=None):
    for state in range(1, len(parent.hb_cut_offs)+1):
        cmd.load_cgo('', '_state_keeper', state)
    parent.display=show_network.ShowNetwork(parent, stb_file, gi_file)
    if stb_file not in ['', None]:
        parent.display.draw_clusters()

    parent.display.make_groups()
    parent.display.draw_constraints(0)
    parent.display.draw_constraints(parent.pymol_state)

    cmd.set_wizard(state_label(parent.hb_cut_offs))

def display_results(parent, mode='overwrite', load_msa=True):

    def exit_display():
        # This will kill already running state_observer after the function is finished.
        # Since this function is not threaded (and can not be threaded because it interacts with gui,
        # the old observer will halt until the execution of this function is complete.
        # However, we start a fresh state_observer using after method, the fresh state observer will only start later;
        # the old state_observer will gain control before that and will eventually be killed because we set parent.kill_state_observer=True.
        # The fresh state observer will set parent.kill_state_observer=False in the beginning
        parent.kill_state_observer=True
        parent.visual_panel.after(int(parent.observ_freq*1000), lambda: state_observer(parent, run_afresh=True))

    if len(parent.li_plot.plot.graph_datas)==parent.dataset_limit:
        print 'The maximum number of data sets that can be showed at the same time is seven'
        exit_display()
        return

    parent.draw_network.config(disabledforeground='black', state='disabled')
    shown_pdb=get_pymol_structure(parent, 'show_pdb_drop', 'File')

    if shown_pdb is None:
        pdb_file=parent.show_pdb.get()
        remove_pdb=False
        if os.path.isfile(pdb_file):
            pdb_obj=read_pdb.ReadPDB(pdb_file)
            if not hasattr(pdb_obj, 'structure'):
                exit_display()
                return
        else:
            errors.input_error(12, args=(pdb_file,))
            exit_display()
            return
    else:
        remove_pdb=True
        pdb_file=get_random_string()+'.pdb'
        # pdb_file=parent.scratch_dir+shown_pdb+'.pdb'
        cmd.save(pdb_file, shown_pdb)
        pdb_obj=read_pdb.ReadPDB(pdb_file)
        if not hasattr(pdb_obj, 'structure'):
            exit_display()
            return

    # Report if pdb_file has missing atoms, non_standard_residues, incomplete_residues, less_hydrogen_residues, no_hydrogen_residues
    pdb_report=pdb_obj.check_missing_atoms()
    if len(pdb_report[1])+len(pdb_report[3])>0:
        errors.input_error(13, args=(pdb_file,))
        exit_display()
        return
    if len(pdb_report[4])>0:
        errors.input_error(22, args=(pdb_file,))
        exit_display()
        return

    # Re-load pdb file into pymol
    cmd.delete('all')
    load_pdb(pdb_file, pdb_obj.pdb_id)

    # Get parameters
    data_dir=parent.open_data_dir.get()
    parameter_file=data_dir+'/parameters.out'
    if os.path.isfile(parameter_file):
        set_cna_parameters(parent, parameter_file)
        li_file=parent.li_plot.open_file.get()
        gi_file=parent.gi_plot.open_file.get()
        unf_file=parent.unf_plot.open_file.get()
        stb_file=parent.stb_plot.open_file.get()
    else:
        li_file=gi_file=unf_file=stb_file=''
        errors.attention(1)
        parent.cna_type='single'
        parent.hb_cut_offs=get_hb_cut_offs([round(float(parent.cna_args[i].get()), 2) for i in ['hb_start', 'hb_stop', 'hb_step']])
        tus_type=int(parent.cna_args['tus_type'].get()[-1])
        if tus_type==1:
            parent.hp_cut_offs=get_hp_cut_offs([round(float(parent.cna_args['hp_start'].get()), 2), None], parent.hb_cut_offs, tus_type)
        else:
            parent.hp_cut_offs=get_hp_cut_offs([round(float(parent.cna_args[i].get()), 2) for i in ['hp_start', 'hp_stop']], parent.hb_cut_offs, tus_type)

    # Get network object
    parent.pdb_obj=pdb_obj
    parent.pdb_file=parent.pdb_obj.pdb_file
    shown_net=parent.show_net_drop.get()
    net_obj=None
    if shown_net=='PDB':
        net_obj=parent.rd.read_pdb_network(parent, pdb_file)
    else:
        shown_file=parent.show_net.get()
        if os.path.isfile(shown_file):
            net_obj=parent.rd.read_constraint_file(parent, shown_file)
        else:
            exit_display()
            print 'Error: Network file does not exist'
            return
    if net_obj is not None:
        parent.net_obj=net_obj
    else:
        net_obj=parent.rd.read_pdb_network(parent, pdb_file)

    if net_obj is not None:
        parent.net_obj=net_obj
    else:
        print 'Error: Failed to generate Network from PDB'
        exit_display()
        return
    if remove_pdb:
        os.remove(pdb_file)
    else:
        parent.pdb_file=pdb_file

    # Overwrite data
    parent.li_plot.datas=[]
    parent.gi_plot.datas=[]
    parent.stb_plot.datas=[]

    # Find out if gi needs to be calculated and we are able to calculate it. Also if clusters are to be drawn
    if parent.cna_exe is not None and not os.path.isfile(gi_file) and os.path.isfile(stb_file) and os.access(os.path.dirname(li_file), os.W_OK):
        calculate_gi=True
    else:
        calculate_gi=False

    stb_file=stb_file if os.path.isfile(stb_file) else ''
    # Load pdb and display network -- This will also calculate gi if needed
    if calculate_gi:
        gi_file=li_file.replace('local', 'global')
        set_field(parent.gi_plot.open_file, gi_file)
        display_network(parent, stb_file, gi_file=gi_file)
    else:
        display_network(parent, stb_file, gi_file=None)

    # Load all plots
    parent.li_plot.load('li', li_file, mode=mode if mode=='overwrite' else 'append')
    parent.gi_plot.load('gi', gi_file, mode=mode if mode=='overwrite' else 'append')
    parent.stb_plot.load(stb_file, mode=mode if mode=='overwrite' else 'append')
    parent.unf_plot.load(unf_file, mode=mode if mode=='overwrite' else 'append')
    detect_table_update(parent)
    parent.state_slider.config(from_=1, to=len(parent.hb_cut_offs))
    cmd.zoom()

    # try to load sequence conservation
    if load_msa:
        msa_file=parent.msa_plot.open_file.get()
        msa_name=parent.msa_plot.basename.get()
        msa_chain=parent.msa_plot.chain_id.get()
        parent.msa_plot.load(msa_file, msa_name, msa_chain, mode='overwrite')

    # TODO: Mode == overwrite clause breaks Visualize data button, not having it breaks compare data, why? i don#t know
    if mode!='compare':
        exit_display()

def mutate(parent, plot_class, chain, resi, src_resname, target_resname, target_frequency):
    # check if the object is loaded in the pymol
    if not parent.pdb_obj.pdb_id in cmd.get_names():
        errors.input_error(26, parent.pdb_obj.pdb_id)
        return

    # kill the state observer observer
    parent.kill_state_observer=True
    # disable all cgo objects and disable objects other than the protein being mutated
    old_frame=cmd.get_frame()
    src_pymol_obj=parent.pdb_obj.pdb_id
    # cgo_objects=cmd.get_names_of_type("object:cgo")
    # for cgo_obj in cgo_objects:
    #    cmd.disable(cgo_obj)
    enabled_objects=cmd.get_names("all", enabled_only=True)
    cmd.disable("all")
    cmd.enable(src_pymol_obj)

    # start mutagenesis wizard
    cmd.wizard("mutagenesis")
    cmd.do("refresh_wizard")
    cmd.get_wizard().set_mode("%s"%target_resname)
    selection="/%s//%s/%s"%(src_pymol_obj, chain, resi)
    cmd.get_wizard().do_select(selection)
    cmd.show('stick', selection)
    cmd.show('stick', 'mutation')
    cmd.zoom("%s around 6"%selection)
    cmd.frame(str(1))

    # Now ask for confirmation
    if not errors.attention(2, (chain, src_resname, resi, target_resname, parent.pdb_obj.pdb_id)):
        cmd.hide('stick', selection)
        cmd.get_wizard().clear()
        cmd.set_wizard()
        for obj in enabled_objects:
            cmd.enable(obj)
        # Now enable state_observer
        parent.visual_panel.after(int(parent.observ_freq*1000), lambda: state_observer(parent, run_afresh=True))

    else:
        # remove cgos
        cgo_objects=cmd.get_names_of_type("object:cgo")
        for cgo_obj in cgo_objects:
            cmd.delete(cgo_obj)

        # save WT if not found
        wt_results_found=len(glob.glob(parent.open_reference_dir.get()+'*local_indices.dat'))>0  # or len(glob.glob(parent.open_reference_dir.get()+'results/*local_indices.dat'))>0
        # make WT directory if not there and results not found, then set the field to this directory
        wt_dir_name=os.path.split(os.path.split(parent.open_reference_dir.get())[0])[-1]
        if not wt_results_found and wt_dir_name!=parent.wt_dir_name:
            wt_dir=parent.open_reference_dir.get()+parent.wt_dir_name+'/'
            if not os.path.isdir(wt_dir):
                os.mkdir(wt_dir)
            set_field(parent.open_reference_dir, wt_dir)
        wt_dir=parent.open_reference_dir.get()
        wt_pdb_file=wt_dir+parent.pdb_obj.pdb_id+'_wt.pdb'

        # try to save wt pdb if results are not found and pdb file is not already there from previous mutation attempts
        if not wt_results_found and not os.path.isfile(wt_pdb_file):
            errors.input_error(25, (parent.open_reference_dir.get()))
            cmd.save(wt_pdb_file, parent.pdb_obj.pdb_id)
            parent.net_obj.write('%s/%s_network.dat'%(wt_dir, parent.pdb_obj.pdb_id))

        # apply mutation now
        cmd.get_wizard().apply()
        cmd.set_wizard()

        # reset pdb object with the mutated pdb
        random_pdb_filename=get_random_string()+'.pdb'
        # following set commands are needed other wise pymol saves pdb where atoms of residues are not together
        cmd.set('retain_order', False)
        cmd.set('pdb_retain_ids', False)
        # cmd.load(random_pdb_filename, parent.pdb_obj.pdb_id, 0)
        cmd.save(random_pdb_filename, parent.pdb_obj.pdb_id)
        cmd.set('retain_order', True)
        cmd.set('pdb_retain_ids', True)
        pdb_obj=read_pdb.ReadPDB(random_pdb_filename, src_pymol_obj)
        if hasattr(pdb_obj, 'structure'):
            parent.pdb_obj=pdb_obj

        # reset msa data and plot
        plot_class.sec_ax.set_title("Substitution frequency for Res %s_%s%d"%(chain, target_resname, resi))
        plot_class.data[plot_class.annotation_x_index]=target_frequency
        plot_class.bars[plot_class.annotation_x_index].set_height(target_frequency)
        plot_class.fig.canvas.draw()

        # reset the network object and display network of the mutant
        net_obj=parent.rd.read_pdb_network(parent, random_pdb_filename)
        if net_obj is not None:
            parent.net_obj=net_obj
            display_network(parent, stb_file=None, gi_file=None)

        # re-enable the objects that were enabled before
        for obj in enabled_objects:
            cmd.enable(obj)

        # reload table
        detect_table_update(parent)

        # remove temporary pdb file
        os.remove(random_pdb_filename)

        # go the the old frame
        cmd.frame(old_frame)

        # Now enable state_observer
        parent.visual_panel.after(int(parent.observ_freq*1000), lambda: state_observer(parent, run_afresh=True))

def run_cna(parent, button, default_label, multi=False):
    if parent.cna_exe is None:
        errors.attention(11)
        return

    parent.run_cna.config(state='disabled')

    if 'again' in parent.run_cna.config()['text'][-1]:
        os.killpg(parent.process_pid[-1], signal.SIGTERM)
        parent.process_log[parent.process_pid[-1]][1]=-1
        parent.run_cna.config(state='normal')
        return

    if 'all' in parent.run_cna.config()['text'][-1]:
        for pid in parent.process_pid:
            os.killpg(pid, signal.SIGTERM)
            parent.process_log[pid][1]=-1
        parent.run_cna.config(state='normal')
        return

    # Catch error in input setting
    parent.run_cna.focus()
    parent.run_cna.update()

    # Get executable arguments
    simulation_type={'SNT Analysis':'--snt', 'ENT Analysis':'--ent', 'FNC Analysis':'--ent_fnc'}[parent.cna_args['cna_type'].get()]
    logfile_verbose={'SNT Analysis':'-v 3', 'ENT Analysis':'-v 1', 'FNC Analysis':'-v 1'}[parent.cna_args['cna_type'].get()]
    thermal_unfolding='--TUS '+parent.cna_args['tus_type'].get()[-1]
    hydrophobic_type='-H '+parent.cna_args['tet_type'].get()[-1]
    akaike_criterion=parent.cna_args['aic'].get()
    transition_source='--transition_source '+parent.cna_args['tran_src'].get().lower().replace(' ', '_')
    unfolding_nuclei=' '.join([parent.cna_args[i].get() for i in ['nuc_1', 'nuc_2', 'nuc_3', 'nuc_4']])
    unfolding_nuclei='--unfolding_nuclei '+unfolding_nuclei if unfolding_nuclei!='   ' else ''
    fnc_network_numb='-F '+parent.cna_args['fnc_num'].get() if simulation_type=='--ent_fnc' else ''
    hydrogen_bond_e='-E '+' '.join([parent.cna_args[i].get() for i in ['hb_stop', 'hb_start', 'hb_step']])
    hydrophobic_cut='-c '+' '.join([parent.cna_args[i].get() for i in ['hp_start', 'hp_stop']][:1 if thermal_unfolding=='--TUS 1' else 2])
    output_settings=' '.join([parent.cna_args[i].get() for i in ['out_stbmap', 'out_network', 'out_ligand', 'all_results']])
    result_directory='--res_dir "'+parent.cna_args['res_dir'].get()[:-1]+'/results"'
    cna_run_log_file=parent.cna_args['res_dir'].get()+'cna_run.log'

    # Remove old log
    if os.path.isfile(cna_run_log_file):
        os.remove(cna_run_log_file)

    # Arrange input pdb(s) and network
    this_dir=os.getcwd()
    work_dir=parent.cna_args['res_dir'].get()
    os.chdir(work_dir)

    if os.path.isdir('results'):
        if errors.attention(7, work_dir):
            os.system('rm results -r')
        else:
            parent.run_cna.config(state='normal')
            return

    if simulation_type in ['--snt', '--ent_fnc']:
        if parent.open_pdb_drop.get()=='File':
            if not os.path.isfile(parent.open_pdb.get()):
                errors.input_error(12, parent.open_pdb.get())
                parent.run_cna.config(state='normal')
                return
            input_structure='-i '+parent.open_pdb.get()
        else:
            selected_save=get_pymol_structure(parent, 'open_pdb_drop', 'File')
            if selected_save is not None:
                cmd.save(work_dir+selected_save+'.pdb', selected_save)
                in_handle=open(work_dir+selected_save+'.pdb', 'r')
                file_data='\n'.join([l for l in in_handle.readlines() if not l.startswith('CONECT')])
                outhandle=open(work_dir+selected_save+'.pdb', 'w')
                outhandle.write(file_data)
                in_handle.close()
                outhandle.close()
                input_structure='-i '+selected_save+'.pdb'
            else:
                parent.run_cna.config(state='normal')
                return
        if parent.open_net_drop.get()=='None':
            input_first_net=''
        elif parent.open_net_drop.get()=='File':
            if not os.path.isfile(parent.open_net.get()):
                errors.input_error(12, parent.open_net.get())
                parent.run_cna.config(state='normal')
                return
            input_first_net='--netin '+parent.open_net.get()
        elif parent.open_net_drop.get()=='PyMOL':
            try:
                network=parent.net_obj.write(work_dir+'/network.dat')
                input_first_net='--netin '+network
            except:
                errors.input_error(6)
                parent.run_cna.config(state='normal')
                return
    elif simulation_type=='--ent':
        input_structure='-i '+parent.open_pdb.get()
        if parent.open_net_drop.get()=='None':
            input_first_net=''
        elif parent.open_net_drop.get()=='File':
            input_first_net='--netin '+parent.open_net.get()
        elif parent.open_net_drop.get()=='PyMOL':
            try:
                network=parent.net_obj.write(work_dir+'/network.dat', cov_only=True)
                input_first_net='--netin '+network
            except:
                errors.input_error(6)
                parent.run_cna.config(state='normal')
                return

    # Get total steps and update frequency of progress bar
    if simulation_type=='--snt':
        total_steps=len(get_hb_cut_offs([round(float(parent.cna_args[i].get()), 2) for i in ['hb_start', 'hb_stop', 'hb_step']]))
        check_freqs=1
    elif simulation_type=='--ent':
        total_steps=len(glob.glob(parent.open_pdb.get()+'*.pdb'))
        check_freqs=10
    elif simulation_type=='--ent_fnc':
        total_steps=int(parent.cna_args['fnc_num'].get())
        check_freqs=1

    cna_exe_str='stdbuf -oL '+parent.cna_exe+' '+' '.join([simulation_type, thermal_unfolding, transition_source, unfolding_nuclei, hydrophobic_type,
                                                           fnc_network_numb, hydrogen_bond_e, hydrophobic_cut, output_settings, akaike_criterion,
                                                           input_structure, input_first_net, result_directory, logfile_verbose, '&>"'+cna_run_log_file+'"'])
    cna_exe_str=' '.join(cna_exe_str.split())

    print 'Initializing CNA run:'
    print cna_exe_str

    # put this in to a subprocess
    def wrap():
        sub=subprocess.Popen(cna_exe_str, shell=True, cwd=work_dir, preexec_fn=os.setsid)
        parent.process_log[sub.pid]=[sub, None]
        parent.process_pid.append(sub.pid)
        if multi:
            parent.process_group.append(sub.pid)

    T=Thread(target=wrap)
    T.start()
    time.sleep(1)
    os.chdir(this_dir)
    progress_observer(parent, parent.process_pid[-1], button, cna_run_log_file, total_steps, check_freqs, default_label, multi)

# Observer functions

def state_observer(parent, run_afresh):

    def clear_all(parent):
        # Destroy network and display in PyMol
        parent.net_obj.destroy()
        parent.display.destroy()
        # Clear all plots
        parent.li_plot.load('li', '', 'overwrite')
        parent.gi_plot.load('gi', '', 'overwrite')
        parent.stb_plot.load('', 'overwrite')
        parent.unf_plot.load('', 'overwrite')
        parent.msa_plot.load('', '', '', 'overwrite')
        # Clear Table
        parent.table_canvas=load_table(parent, parent.table_frame)
        parent.table_canvas.model.createEmptyModel()
        # Destroy PDB object
        parent.pdb_obj=None
        # Remove E-cut-off display
        cmd.set_wizard()

    def detect_network_update(parent):
        # Function to get constraints early or late, in case of prompting while constraints are being drawn
        def get_constraints(early=True):
            try:
                return cmd.get_names_of_type('object:cgo')
            except:
                if early:
                    return parent.pymol_const
                else:
                    time.sleep(parent.observ_freq)
                    return cmd.get_names_of_type('object:cgo')

        # Detect creation and deletion of constraints
        old_pymol_state=cmd.get_state()
        old_pymol_const=get_constraints()
        time.sleep(parent.observ_freq)
        new_pymol_state=cmd.get_state()
        if new_pymol_state!=old_pymol_state: return
        new_pymol_const=get_constraints(False)

        if new_pymol_state==old_pymol_state and new_pymol_const!=old_pymol_const:
            parent.pymol_const=new_pymol_const
            created_constraints=[label for label in new_pymol_const if not label in old_pymol_const and not any([i in label for i in ['_state_keeper', 'RC']])]
            deleted_constraints=[label for label in old_pymol_const if not label in new_pymol_const and not any([i in label for i in ['_state_keeper', 'RC']])]
            # Delete or create constraints
            if deleted_constraints!=[]:
                parent.net_obj.delete_constraints(deleted_constraints)
            if created_constraints!=[]:
                parent.net_obj.create_constraints([created_constraints[0]])

    def detect_plot_update(parent):
        # Update local index and stb_map plot if pdb objects are consistent
        # Get Selected Residues, Existing Constraints and Last Clicked Residue
        cur_pymol_selec=[]
        def append_selected_resi(chain, resi):
            pymol_selector="%s/%s/"%(chain, resi)
            if pymol_selector not in cur_pymol_selec:
                cur_pymol_selec.append(pymol_selector)

        for selection in cmd.get_names('selections'):
            cmd.iterate(selection, "append_selected_resi(chain, resi)", space={'append_selected_resi':append_selected_resi})

        resi_last_click=[i for i in cur_pymol_selec if not i in parent.select_resi]
        resi_last_click=resi_last_click[-1] if resi_last_click!=[] else parent.pymol_resid
        update=False
        if cur_pymol_selec!=parent.select_resi:
            parent.select_resi=cur_pymol_selec
            update=True
        if resi_last_click!=parent.pymol_resid:
            parent.pymol_resid=resi_last_click
            update=True
        if update and parent.pymol_resid in parent.select_resi:
            try:
                parent.select_resi=cur_pymol_selec
                parent.pymol_resid=resi_last_click
                x_index=parent.pdb_obj.pymol_selector_resi_serial[resi_last_click]-1
                parent.li_plot.plot.update_annotation(x_index=x_index)
                parent.unf_plot.plot.update_annotation(x_index=x_index)
                parent.msa_plot.plot.update_annotation(x_index=x_index)
                parent.msa_plot.plot.sec_update(x_index=x_index)

                if len(parent.select_resi)==2:
                    x_index=parent.pdb_obj.pymol_selector_resi_serial[max(parent.select_resi)]-1
                    y_index=parent.pdb_obj.pymol_selector_resi_serial[min(parent.select_resi)]-1
                    parent.stb_plot.plot.update_annotation(x_index, y_index)
            except:
                pass

    def detect_state_update(parent):
        parent.pymol_state=cmd.get_state()
        parent.gi_plot.plot.update_annotation(x_index=parent.pymol_state-1)
        parent.li_plot.plot.line_marker(parent.hb_cut_offs[parent.pymol_state-1])
        parent.display.kill_old_observers=False
        parent.display.draw_constraints(parent.pymol_state)
        cmd.refresh_wizard()
        try:
            cmd.enable("sele")
        except:
            pass

    if run_afresh:
        parent.kill_state_observer=False
        parent.draw_network.config(disabledforeground='black', state='normal')
    if parent.display.kill_old_observers or parent.kill_state_observer:
        return

    if parent.pdb_obj.pdb_id not in cmd.get_names('all') and hasattr(parent, 'net_obj'):
        if errors.attention(8, parent.pdb_obj.pdb_id):
            clear_all(parent)
            return
        else:
            random_pdb_filename=get_random_string()+'.pdb'
            parent.pdb_obj.write_pdb_file(random_pdb_filename)
            load_pdb(random_pdb_filename, parent.pdb_obj.pdb_id)
            os.remove(random_pdb_filename)

    parent.display.kill_old_observers=True
    detect_plot_update(parent)
    if cmd.get_state()!=parent.pymol_state:
        detect_state_update(parent)
    else:
        detect_network_update(parent)
    parent.state_slider.set(parent.pymol_state)
    parent.display.kill_old_observers=False

    if parent.display.kill_old_observers or parent.kill_state_observer:
        return
    parent.visual_panel.after(int(parent.observ_freq*100), lambda:state_observer(parent, run_afresh=False))

def progress_observer(parent, pid, button, status_file, total_steps, check_freqs, default_label='RUN CNA', multi=False):

    def change_button(state=None):
        label=' '.join(button.config()['text'][-1])
        if state=='finished':
            button.config(state='disabled', text='Finished: Writing results...')
        elif label==default_label and pid in parent.process_pid:
            button.config(state='disabled', text='Running: 0 % Complete')
        elif state=='done':
            button.config(state='normal', text=default_label)
        elif 'Complete' in label:
            if multi:
                button.config(state='normal', text='Running: '+str(display_progress)+' % Complete [Click to cancel all jobs]')
            else:
                button.config(state='normal', text='Running: '+str(display_progress)+' % Complete [Click again to cancel]')

    def reset():
        parent.process_log.pop(pid, None)
        parent.process_pid.pop(parent.process_pid.index(pid))
        change_button(state='done')

    try:
        (process, progress)=parent.process_log[pid]
    except:
        return

    if progress is not None and progress<0:
        print 'CNA Job terminated, result files may be incomplete.'
        reset()
        return

    progress=0
    display_progress=progress
    if not os.path.isfile(status_file):
        change_button()
    else:
        error=int(subprocess.check_output('grep -i error "%s" | wc'%status_file, shell=True).split()[0])!=0
        greps=float(subprocess.check_output('grep -i dilution "%s" | wc'%status_file, shell=True).split()[0])
        progress=round((greps)/total_steps*100, 1) if greps>0 else 0
        display_progress=progress
        if multi:
            group=parent.process_pid if parent.process_group==[] else parent.process_group
            multi_progress=[parent.process_log[i][1] for i in parent.process_log if i in group and i!=pid]+[progress]
            display_progress=round(sum([i for i in multi_progress if i is not None])/float(len(multi_progress)), 1)

        if error:
            errors.input_error(20, status_file)
            reset()
            return

        # Update the button text
        change_button()
        if display_progress>=100:
            progress=None
            change_button('finished')
            if process.poll() is not None:
                reset()
                save_dir=os.path.split(status_file)[0]+'/results/'
                if not multi and errors.attention(0, save_dir):
                    parent.panels.selectpage(parent.panel_names['visual_panel'])
                    set_field(parent.open_data_dir, save_dir)
                    detect_data_dir_update(parent, save_dir)
                    display_results(parent)
                    return
                else:
                    return
            elif process.pid<0:
                errors.input_error(20, status_file)
                reset()
                return

    parent.process_log[pid]=[process, progress]
    parent.analyze_panel.after(check_freqs*1000, lambda:progress_observer(parent, pid, button, status_file, total_steps, check_freqs, default_label, multi))

def compare_observer(parent, reference_field, comparison_field, reference_results_found=False, comparison_results_found=False):
    jobs_finished=all([i not in parent.process_log for i in parent.process_group])
    all_results_found=reference_results_found and comparison_results_found

    def update_fields():
        if os.path.isdir(reference_field.get()+'results/'):
            set_field(reference_field, reference_field.get()+'results/')
        if os.path.isdir(comparison_field.get()+'results/'):
            set_field(comparison_field, comparison_field.get()+'results/')

    if all_results_found or jobs_finished:
        if not all_results_found:
            if not errors.attention(0, reference_field.get()+' & '+comparison_field.get()):
                update_fields()
                return

        # Test that mutant and reference data match
        try:
            ref_residues=len(read_pdb.ReadPDB(glob.glob(parent.open_reference_dir.get()+'*.pdb')[0]).res_objects)
            mut_residues=len(read_pdb.ReadPDB(glob.glob(parent.open_comparison_dir.get()+'*.pdb')[0]).res_objects)
            if ref_residues!=mut_residues:
                errors.input_error(28)
                return
        except:
            print 'Missing input pdb file for comparison'
            return

        old_data_dir=parent.open_data_dir.get()

        # Load reference data
        if reference_results_found:
            set_field(parent.open_data_dir, reference_field.get())
        else:
            set_field(parent.open_data_dir, reference_field.get()+'results/')
        detect_data_dir_update(parent, parent.open_data_dir.get())
        display_results(parent, mode='overwrite', load_msa=False)

        # Load comparison data
        if comparison_results_found:
            set_field(parent.open_data_dir, comparison_field.get())
        else:
            set_field(parent.open_data_dir, comparison_field.get()+'results/')
        detect_data_dir_update(parent, parent.open_data_dir.get())
        display_results(parent, mode='compare', load_msa=False)

        # set data dir to the old directory and go to visualize panel
        set_field(parent.open_data_dir, old_data_dir)
        update_fields()

        parent.panels.selectpage(parent.panel_names['visual_panel'])
        return

    parent.mutate_panel.after(1000, lambda:compare_observer(parent, reference_field, comparison_field, reference_results_found, comparison_results_found))













