"""
    Created Mar 2014
    This file is part of VisualCNA
    Copyright (2014) Prakash Chandra Rathi and Daniel Mulnaes

    VisualCNA is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    VisualCNA is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
#####################################################################
# -*- coding: utf-8 -*-
#
# Authors: Daniel Mulnaes and Prakash Chandra Rathi
# Heinrich Heine University, Duesseldorf
# Institute for Pharmaceutical and Medicinal Chemistry
# Universitaetsstr. 1 40225 Duesseldorf
# Germany
#####################################################################


# External modules
from pymol import cmd

# VisualCNA modules
import methods, errors

class Network():
    def __init__(self, parent):
        self.constraints={}
        self.label_to_id={}
        self.id_to_label={}
        self.action_logs=[]
        self.undone_logs=[]
        self.pdb_object=parent.pdb_obj
        self.hb_cut_offs=parent.hb_cut_offs
        self.hp_cut_offs=parent.hp_cut_offs
        self.parent=parent

        self.colors={'HBOND':'red',
                       'SBRIDGE':'magenta',
                       'HPHOBES':'green',
                       'SRING':'blue',
                       'DBRIDGE':'yellow',
                       'COV':'black',
                       'CUSTOM':'black',
                       'LIG_COV': 'black'}
        self.long_to_short={'HBOND':'HB',
                            'SBRIDGE':'SB',
                            'HPHOBES':'HP',
                            'SRING':'SR',
                            'DBRIDGE':'DB',
                            'COV':'CV',
                            'CUSTOM':'CV',
                            'LIG_COV':'CV'}
        self.short_to_long={'HB':'HBOND',
                            'SB':'SBRIDGE',
                            'HP':'HPHOBES',
                            'SR':'SRING',
                            'DB':'DBRIDGE',
                            'CV':'CUSTOM'}
        self.short_to_text={'HB':'Hydrogen Bond',
                            'SB':'Salt Bridge',
                            'HP':'Hxdrophobic Tether',
                            'SR':'Stacked Ring',
                            'DB':'Disulfide Bridge',
                            'CV':'Covaelent Bond'}


        if self.hb_cut_offs is None:
            self.state_to_net={0:{'create':[], 'delete':[]}}
        else:
            self.state_to_net={i:{'create':[], 'delete':[]} for i in range(len(self.hb_cut_offs)+1)}

    def get_label(self, ctype, atom_id_1, atom_id_2):
        ctype=self.long_to_short[ctype]
        atom1_detail=self.pdb_object.atom_details[atom_id_1]
        atom2_detail=self.pdb_object.atom_details[atom_id_2]
        desc1=methods.get_single_resi_codes(atom1_detail[2])+str(atom1_detail[4])+atom1_detail[3].replace(' ', '_')+'.'+atom1_detail[1]
        desc2=methods.get_single_resi_codes(atom2_detail[2])+str(atom2_detail[4])+atom2_detail[3].replace(' ', '_')+'.'+atom2_detail[1]
        return ctype+'-'+desc1+'-'+desc2

    def add(self, ctype, bars, energy, atom_id_1, atom_id_2):
        color=self.colors[ctype]
        label=self.get_label(ctype, atom_id_1, atom_id_2)
        if label in self.label_to_id:
            print 'This Constraint already Exists'
            return False

        self.constraints[(atom_id_1, atom_id_2)]={'type':ctype, 'bars':bars, 'energy':energy, 'label':label, 'color':color, 'status':0}
        self.label_to_id[label]=(atom_id_1, atom_id_2)
        self.id_to_label[(atom_id_1, atom_id_2)]=label

        # Add Hydrogen Bonds
        if ctype in ['HBOND', 'SBRIDGE']:
            for (cut, state) in zip(self.hb_cut_offs, range(1, len(self.hb_cut_offs)+1)):
                if self.constraints[(atom_id_1, atom_id_2)]['energy']<cut:
                    self.state_to_net[state]['create'].append((atom_id_1, atom_id_2))
                else:
                    self.state_to_net[state]['delete'].append((atom_id_1, atom_id_2))
        # Add Hydrophobic tethers TUS2
        elif ctype=='HPHOBES' and len(self.hp_cut_offs)>1:
            for (cut, state) in zip(self.hp_cut_offs, range(1, len(self.hp_cut_offs)+1)):
                if self.constraints[(atom_id_1, atom_id_2)]['energy']<cut:
                    self.state_to_net[state]['create'].append((atom_id_1, atom_id_2))
                else:
                    self.state_to_net[state]['delete'].append((atom_id_1, atom_id_2))
        # Add Hydrophobic tethers TUS1 and other constraints
        elif not ctype in ['COV', 'LIG_COV']:
            self.state_to_net[0]['create'].append((atom_id_1, atom_id_2))

        return True

    def delete(self, label):
        constraint=self.label_to_id[label]
        if self.constraints[constraint]['status']!=-1:
            self.constraints[constraint]['status']=-1
            print 'Deleted constraint: %s'%label
            return True
        else:
            return False

    def create(self, label, bars):
        constraint=self.label_to_id[label]
        if self.constraints[constraint]['status']==-1:
            self.constraints[constraint]['status']=0
            self.constraints[constraint]['bars']=bars
            print 'Created constraint: %s with %d bars'%(label, bars)
            return True
        else:
            return False

    def delete_constraints(self, labels, log_update=True, table_update=True):
        deleted=[self.delete(l) for l in labels]
        if any(deleted):
            self.parent.display.force_network_draw(labels)
            if log_update:   self.action_logs.append(('d', labels))
            if table_update: self.parent.table_canvas.redrawTable()

    def create_constraints(self, labels, log_update=True, table_update=True):
        bars=[self.constraints[self.label_to_id[l]]['bars'] for l in labels]
        created=[self.create(l, bars[i]) for i, l in enumerate(labels)]
        if any(created):
            self.parent.display.force_network_draw(labels)
            if log_update: self.action_logs.append(('c', labels))
            if table_update: self.parent.table_canvas.redrawTable()

    def add_constraint(self, label, bars, atom_id_1=None, atom_id_2=None):
        if label[:2] in ['HB', 'SB', 'DB', 'SR', 'HP']:
            if label in self.label_to_id:
                self.create_constraints([label])
            else:
                errors.input_error(27, (atom_id_1, atom_id_2, self.short_to_text[label[:2]]))
        elif label[:2] in 'CV' and atom_id_1 is not None and atom_id_2 is not None:
            (atom_id_1, atom_id_2)=sorted([atom_id_1, atom_id_2])
            atom1_cord=self.pdb_object.atom_details[atom_id_1][-3:]
            atom2_cord=self.pdb_object.atom_details[atom_id_2][-3:]
            const_dist=round(sum([(atom1_cord[i]-atom2_cord[i])**2 for i in range(3)])**0.5, 1)

            if const_dist>=5.0 and not errors.attention(10, (atom_id_1, atom_id_2, const_dist)):
                return

            if self.add('CUSTOM', bars, None, atom_id_1, atom_id_2):
                self.action_logs.append(('c', [label]))
                print 'Added constraint: %s with %d bars'%(label, bars)
                self.parent.display.force_network_draw([label])
                methods.detect_table_update(self.parent)

    def undo(self):
        try:
            (action, constraints)=self.action_logs.pop()
            self.undone_logs.append((action, constraints))
            if action=='d':
                self.create_constraints(constraints, log_update=False)
            elif action=='c':
                self.delete_constraints(constraints, log_update=False)
        except:
            print 'No changes to undo'

    def redo(self):
        try:
            (action, constraints)=self.undone_logs.pop()
            self.action_logs.append((action, constraints))
            if action=='c':
                self.create_constraints(constraints, log_update=False)
            elif action=='d':
                self.delete_constraints(constraints, log_update=False)
        except:
            print 'No changes to redo'

    def get_table(self, ctype):
        table_format=lambda c, d: {'Atom 1':c[0], 'Atom 2':c[1], 'Bars':d['bars'], 'Energy':'None' if d['energy'] is None else round(d['energy'], 2)}
        bond_table={}
        if ctype in ['HBOND', 'SBRIDGE']:
            for c in self.constraints:
                if self.constraints[c]['type']==ctype and self.constraints[c]['energy']<max(self.hb_cut_offs):
                    bond_table[self.constraints[c]['label']]=table_format(c, self.constraints[c])
        # CUSTOM in case of covalent bonds show ligand as well as disulfide bridges
        elif ctype=='CUSTOM':
            for c in self.constraints:
                if self.constraints[c]['type'] in ['CUSTOM', 'LIG_COV', 'DBRIDGE']:
                    bond_table[self.constraints[c]['label']]=table_format(c, self.constraints[c])
        else:
            for c in self.constraints:
                if self.constraints[c]['type']==ctype:
                    bond_table[self.constraints[c]['label']]=table_format(c, self.constraints[c])

        return bond_table

    def format_bond_type(self, ctype, bond_set, description, filename):
        bond_list=[(c, self.constraints[c]) for c in self.constraints if self.constraints[c]['type'] in bond_set and self.constraints[c]['status']!=-1]
        # Sort bonds according to atom_id's in case of covaelents and energy otherwise
        if ctype=='COV':
            bond_list=sorted(bond_list, key=lambda k: k[0])
        else:
            bond_list=sorted(bond_list, key=lambda k: k[1]['energy'], reverse=True)

        # Write constraint type header
        outputstring='#'*80+'\n#\n'
        outputstring+='# %s in structure %s identified by FIRST\n'%(description, filename)
        outputstring+='#\n'+'#'*80+'\n'
        atom_1_label='HYDROGEN' if ctype=='HBOND' else 'ATOM 1'
        atom_2_label='ACCEPTOR' if ctype=='HBOND' else 'ATOM 2'
        energy_label='ENERGY'   if ctype=='HBOND' else 'DISTANCE'
        header_args=('#TYPE', atom_1_label, atom_2_label, 'BARS', energy_label, 'RECORD 1', 'RECORD 2', 'HB TYPE')

        # Format a single constraint as a string line
        def format_line(args, head=False):
            (a, b, c, d, e, f, g, h)=args
            if ctype=='COV':
                if head: return '%-10s%10s%10s%15s%20s%10s\n'%(a, b, c, d, f, g)
                else:    return '%-10s%10i%10i%15i%20s%10s\n'%(a, b, c, d, f, g)
            if ctype=='HBOND':
                if head: return '%-10s%10s%10s%10s%15s%10s%10s%10s\n'%(a, b, c, d, e, f, g, h)
                else:    return '%-10s%10i%10i%10i%15.8f%10s%10s%10s\n'%(a, b, c, d, e, f, g, h)
            if ctype in ['HPHOBES', 'SRING']:
                if head: return '%-10s%10s%10s%10s%15s%10s%10s\n'%(a, b, c, d, e, f, g)
                else:    return '%-10s%10i%10i%10i%15.3f%10s%10s\n'%(a, b, c, d, e, f, g)

        outputstring+=format_line(header_args, True)

        if len(bond_list)==0 and ctype=='SRING':
            outputstring+='# No stacked aromatic rings identified by FIRST\n'
            return outputstring

        # Format all constraints of given type
        for bond in bond_list:
            try:    h={'HBOND':'HB', 'SBRIDGE':'SB'}[bond[1]['type']]
            except: h=''
            body_args=(ctype, bond[0][0], bond[0][1], bond[1]['bars'], bond[1]['energy'],
                         self.pdb_object.atom_details[bond[0][0]][0], self.pdb_object.atom_details[bond[0][1]][0], h)
            outputstring+=format_line(body_args)

        return outputstring

    def write(self, filename, cov_only=False):
        net=open(filename, 'w')
        net.write(self.format_bond_type('COV', ['COV', 'LIG_COV', 'CUSTOM', 'DBRIDGE'], 'Covalent interactions', filename))
        if not cov_only:
            net.write(self.format_bond_type('HBOND', ['HBOND', 'SBRIDGE'], 'Hydrogen bonds', filename))
            net.write(self.format_bond_type('HPHOBES', ['HPHOBES'], 'Hydrophobic tethers', filename))
            net.write(self.format_bond_type('SRING', ['SRING'], 'Aromatic stacking interactins', filename))
        net.close()
        return filename

    def destroy(self):
        for label in self.label_to_id:
            try:
                cmd.delete(label)
            except:
                pass

        del self.parent.net_obj
