"""
    Created Mar 2014
    This file is part of VisualCNA
    Copyright (2014) Prakash Chandra Rathi and Daniel Mulnaes

    VisualCNA is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    VisualCNA is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
#####################################################################
# -*- coding: utf-8 -*-
#
# Authors: Prakash Chandra Rathi and Daniel Mulnaes
# Heinrich Heine University, Duesseldorf
# Institute for Pharmaceutical and Medicinal Chemistry
# Universitaetsstr. 1 40225 Duesseldorf
# Germany
#####################################################################


# Standard python libraries
import os
from Tkinter import *
import Tkinter as Tk

# External modules
import Pmw
import matplotlib
from matplotlib.ticker import MaxNLocator, FuncFormatter
from matplotlib.backend_bases import NavigationToolbar2
from matplotlib.backends.backend_tkagg import FigureCanvasTkAgg, NavigationToolbar2TkAgg
from matplotlib.figure import Figure
from pymol import cmd
import numpy as np
import pylab

# VisualCNA Modules
import widgets, errors, methods

# TODO: Make titles "No file loaded for line and image plots"
class MyCustomToolbar(NavigationToolbar2TkAgg, Tk.Frame):
    def __init__(self, canvas, window, parent):
        self.parent=parent
        self.canvas=canvas
        self.window=window
        self._idle=True
        self.tooltip_dict={'Home':self.parent.tooltips.plot_home,
                           'Pan':self.parent.tooltips.plot_pan,
                           'Zoom':self.parent.tooltips.plot_zoom,
                           'Save':self.parent.tooltips.plot_save}
        NavigationToolbar2.__init__(self, canvas)

    # disable mouse movement feedback, otherwise it changes the size of the toolbar
    def destroy(self, *args):
        Tk.Frame.destroy(self, *args)

    def mouse_move(self, event):
        pass
    # disable message
    def set_message(self, s):
        pass

    # keep button pressed
    def set_active(self, ind):
        self._ind=ind
        self._active=[self._axes[i] for i in self._ind]

    def _Button(self, text, file, command):

        def activate(c):
            text=b.config()['text'][-1]
            relief=b.config()['relief'][-1]

            if text in ["Home", "Save"]:
                c()
                return

            for tbb in self.toolbar_buttons:
                tbb_text=tbb.config()['text'][-1]
                tbb_relief=tbb.config()['relief'][-1]

                if tbb_text!=text:
                    tbb.config(relief='raised')
                else:
                    if relief=='raised':
                        b.config(relief='sunken')
                    elif relief=='sunken':
                        b.config(relief='raised')
                    c()
        rcParams=matplotlib.rcParams
        file=os.path.join(rcParams['datapath'], 'images', file)
        im=Tk.PhotoImage(master=self, file=file)
        b=Tk.Button(
            master=self, text=text, padx=2, pady=2, image=im, command=lambda c=command:activate(c))
        b._ntimage=im
        b.pack(side=Tk.LEFT)
        Pmw.Balloon(self).bind(b, balloonHelp=self.tooltip_dict[text])
        return b



    def _init_toolbar(self):
        xmin, xmax=self.canvas.figure.bbox.intervalx
        height, width=50, xmax-xmin
        Tk.Frame.__init__(self, master=self.window,
                          width=int(width), height=int(height),
                          borderwidth=2)

        self.update()  # Make axes menu

        self.bHome=self._Button(text="Home", file="home.ppm",
                                   command=self.home)

        self.bPan=self._Button(text="Pan", file="move.ppm",
                                  command=self.pan)

        self.bZoom=self._Button(text="Zoom",
                                   file="zoom_to_rect.ppm",
                                   command=self.zoom)

        self.bsave=self._Button(text="Save", file="filesave.ppm",
                                   command=self.save_figure)
        self.toolbar_buttons=[self.bHome, self.bPan, self.bZoom, self.bsave]
        # self.message = Tk.StringVar(master=self)
        # self._message_label = Tk.Label(master=self, textvariable=self.message)
        # self._message_label.pack(side=Tk.RIGHT)
        self.pack(side=Tk.BOTTOM, fill=Tk.X)


class line_plot:

    def __init__(self, parent, plot_frame, toolbar_frame, y_types, x_colname, x_lab, y_lab, title, reverse_x=False, graph_datas=[], transition_datas=[]):
        # Variables that need to be changed when loading new data
        self.parent=parent
        self.x_colname=x_colname
        self.graph_datas=graph_datas
        self.transition_datas=transition_datas
        self.x_values=self.graph_datas[0][self.x_colname] if self.graph_datas!=[] else []

        # Fixed plot variables
        self.reverse_x=reverse_x
        self.y_types=y_types
        self.title=title
        parent_width=int(plot_frame.config()["width"][-1])
        parent_height=int(plot_frame.config()["height"][-1])
        self.size_conversion=1/float(pylab.gcf().get_dpi())
        self.fig=Figure(figsize=(parent_width*self.size_conversion, parent_height*self.size_conversion))
        self.line_pattern=['-', '--', ':', '-.', 'steps--']
        self.line_colors=["black", "red", "green", "yellow", "magenta", "cyan", "blue"]
        self.mark_color='r'
        self.annotation_x_index=0
        self.annotated_line=None

        # Set labels and title
        self.ax=self.fig.add_subplot(111)
        self.x_lab=x_lab
        self.y_lab=y_lab
        self.ax.set_xlabel(self.x_lab)
        self.ax.set_ylabel(self.y_lab)
        if self.graph_datas==[]:
            self.ax.set_title("No %s loaded"%self.title)
        else:
            self.ax.set_title(self.title)

        # Set layout
        self.fig.tight_layout()
        self.pick_dict={'d_min':100}
        self.pick_numb=0
        self.lines=[]
        self.transitions=[]
        self.index_to_draw=[]
        self.files_to_draw=[]

        # Set marker and annotation
        if not self.reverse_x:
            self.hline=self.ax.hlines(max(self.parent.hb_cut_offs),
                                      0,  # min(self.x_values),
                                      0,  # max(self.x_values),
                                      colors=self.mark_color,
                                      linestyles='solid')
            self.ax.xaxis.set_major_locator(MaxNLocator(5))
            self.ax.xaxis.set_major_formatter(FuncFormatter(self.__resid_formatter))

        self.circle=self.ax.plot(None, None, color=self.mark_color, markeredgecolor=self.mark_color, marker='.', markersize=12.0)[0]
        self.annotation=self.ax.text(-0.1,-0.125, '',
                                     horizontalalignment='left',
                                     verticalalignment='top',
                                     transform=self.ax.transAxes)

        # Set Canvas
        canvas=FigureCanvasTkAgg(self.fig, master=plot_frame)
        canvas.get_tk_widget().pack(side=TOP, fill=BOTH, expand=0)
        canvas.mpl_connect('pick_event', self.on_click)

        # Set Toolbar
        toolbar=MyCustomToolbar(canvas, toolbar_frame, self.parent)
        toolbar.update()
        toolbar.pack(side=TOP, fill=BOTH, expand=1)

    def __resid_formatter(self, x, pos):
        'The two args are the value and tick position'
        if x==0:
            return str(self.parent.pdb_obj.res_objects[int(x)].id[1]-1)
        try:
            return self.parent.pdb_obj.resi_serial_pymol_selector[x].replace('/', '')
        except:
            return ''

    def clear(self):
        # Remove previously drawn lines
        for group in self.lines:
            for line in group:
                self.ax.lines.remove(line)
        if self.reverse_x:
            for group in self.transitions:
                for line in group:
                    line.remove()
        self.lines=[]
        self.transitions=[]

    def add_data(self, graph_datas, transition_datas):
        self.graph_datas.append(graph_datas)
        self.transition_datas.append(transition_datas)
        self.x_values=self.graph_datas[0][self.x_colname]

    def reload(self, parent):
        if self.graph_datas==[]:
            self.parent=parent
            self.index_to_draw=[]
            self.files_to_draw=[]
            self.clear()
            # Change title
            self.ax.set_title("No %s loaded"%self.title)
            self.x_values=[]
            self.y_values=[]
            self.update_annotation(0, 0)
            self.fig.canvas.draw()
        else:
            # Overwrite parent and xvalues
            self.parent=parent
            self.x_values=self.graph_datas[0][self.x_colname]
            # Reset horizontal line
            if not self.reverse_x:
                y_coord=self.hline.get_paths()[0].vertices[0][1]
                self.hline.set_paths([((min(self.x_values), y_coord), (max(self.x_values), y_coord))])
            # Reset title
            self.ax.set_title(self.title)

    def remake_plot(self):
        # Set data and marker
        x_coord=self.x_values[self.annotation_x_index]
        if self.lines!=[]:
            y_coord=self.lines[0][0].get_data()[1][self.annotation_x_index]
            self.update_annotation(x_coord=x_coord, y_coord=y_coord)
        else:
            self.update_annotation(x_index=self.annotation_x_index)

    def update_annotation(self, x_coord=None, y_coord=None, x_index=None):
        if self.lines==[]:
            return
        if x_index is not None:
            x_coord=self.x_values[x_index]
            y_coord=self.annotated_line.get_data()[1][x_index]

        if not self.reverse_x and x_index is None:
            self.line_marker(y_coord=y_coord)

        if self.reverse_x:
            annotation_text="Pos: %.2f, %.2f"%(x_coord, y_coord)
        else:
            annotation_text="Pos: %s, %.2f"%(self.__resid_formatter(x_coord, None), y_coord)
        self.annotation.set_text(annotation_text)
        self.circle.set_data([x_coord, y_coord])
        self.fig.canvas.draw()

    def line_marker(self, y_coord):
        if self.lines==[]:
            return
        self.hline.set_paths([((min(self.x_values), y_coord), (max(self.x_values), y_coord))])
        self.fig.canvas.draw()

    def line_picker(self, artist, mouseevent):
        """
        find the points within a certain distance from the mouseclick in
        graph_datas coords and attach some extra attributes, pickx and picky
        which are the graph_datas points that were picked
        """
        if mouseevent.xdata is None:
            return False, dict()
        d_threshold=20  # maximum distance in pixels
        fig_height=artist.get_figure().get_figheight()*artist.get_figure().get_dpi()
        fig_width=artist.get_figure().get_figwidth()*artist.get_figure().get_dpi()

        x_data=artist.get_xdata()
        y_data=artist.get_ydata()
        x_range=abs(max(x_data)-min(x_data))
        y_range=abs(max(y_data)-min(y_data))
        d_pixel_x=abs(mouseevent.xdata-x_data)*fig_width/x_range
        d_pixel_y=abs(mouseevent.ydata-y_data)*fig_height/y_range
        d_pixel=np.sqrt((d_pixel_x**2.0)+(d_pixel_y**2.))
        x_index=np.argmin(d_pixel)
        d_min=d_pixel[x_index]
        x_pick=x_data[x_index]
        y_pick=y_data[x_index]
        if d_min<=d_threshold and d_min<self.pick_dict['d_min']:
            self.pick_dict=dict(d_min=d_min, ind=x_index, pickx=x_pick, picky=y_pick, uniq_artist=artist)
        self.pick_numb+=1

        shown_lines=self.files_to_draw.count(1)*[x[1] for x in self.index_to_draw].count(1)
        if self.pick_numb==shown_lines:
            self.pick_numb=0
            return_values=self.pick_dict
            self.pick_dict={'d_min':100}
            if return_values.has_key('ind'):
                return True, return_values
        # If not all the lines are checked
        return False, dict()

    def on_click(self, event):
        x_pick=event.pickx
        y_pick=event.picky
        self.annotation_x_index=event.ind
        self.annotated_line=event.uniq_artist
        if self.reverse_x:
            state=self.annotation_x_index+1
        else:
            y_val=round(y_pick, 2)
            state=self.parent.hb_cut_offs.index(y_val)+1 if y_val!=0.0 else 1
            # select and zoom on the residue
            try:
                selection=self.parent.pdb_obj.resi_serial_pymol_selector[self.annotation_x_index+1]+" and %s"%self.parent.pdb_obj.pdb_id
                zoom_selection="(%s and %s) around 5"%(selection, self.parent.pdb_obj.pdb_id)
                cmd.select("sele", selection)
                cmd.zoom(zoom_selection)
                self.parent.pymol_resid=self.annotation_x_index+1
            except: pass

        self.update_annotation(x_coord=x_pick, y_coord=y_pick)
        cmd.frame(state)

    def draw_lines(self, index_to_draw, display):

        self.index_to_draw=index_to_draw
        self.files_to_draw=[d.get() for d in display]
        self.clear()

        if self.graph_datas==[]:
            return

        # Find minimum and maximum x and y values to determines axes
        self.xstremes=(min([min(data[self.x_colname]) for data in self.graph_datas]), max([max(data[self.x_colname]) for data in self.graph_datas]))
        try:
            self.ystremes=(min([min(data[y]) for f, data in enumerate(self.graph_datas) if self.files_to_draw[f] for y in [i for (i, d) in self.index_to_draw if d]]),
                           max([max(data[y]) for f, data in enumerate(self.graph_datas) if self.files_to_draw[f] for y in [i for (i, d) in self.index_to_draw if d]]))
        except:
            self.ystremes=(0,)

        # Draw new lines
        drawn=False
        colno=0
        # For each file to be drawn
        for file_index, draw in enumerate(self.files_to_draw):
            if not draw:
                continue

            file_lines=[]
            file_transitions=[]
            # For each index to be drawn
            for i, index_name in enumerate([index_name for (index_name, draw_index) in self.index_to_draw if draw_index]):
                patterno=range([d[1] for d in self.index_to_draw].count(1))[i]
                y_values=self.graph_datas[file_index][index_name]
                # Make line-segment for graph
                file_lines.append(self.ax.plot(
                                  self.x_values,
                                  y_values,
                                  picker=lambda a, m: self.line_picker(a, m),
                                  color=self.line_colors[colno],
                                  linestyle=self.line_pattern[patterno],
                                  label=index_name)[0])
                drawn=True

                # Make line-segment for transition if found
                if self.reverse_x:
                    try:
                        file_transitions.append(self.ax.vlines(
                                      self.transition_datas[file_index][index_name],
                                      min(self.ystremes),
                                      max(self.ystremes),
                                      color=self.line_colors[colno],
                                      linestyle=self.line_pattern[patterno]))
                    except:
                        # for MCS, transition is not calculated
                        pass
            colno+=1
            self.lines.append(file_lines)
            self.transitions.append(file_transitions)

        # Set legend
        handles, labels=self.ax.get_legend_handles_labels()
        leg=self.ax.legend(handles, labels, loc='best', prop=dict(size=10))
        leg.get_frame().set_alpha(0)

        # Update Axis & Annotation
        if drawn:
            self.annotated_line=self.lines[0][0]
            # Calculate and set axis margins
            self.x_margin=(max(self.xstremes)-min(self.xstremes))*0.04
            self.y_margin=(max(self.ystremes)-min(self.ystremes))*0.04
            self.ax.set_xlim([min(self.xstremes)-self.x_margin, max(self.xstremes)+self.x_margin])
            self.ax.set_ylim([min(self.ystremes)-self.y_margin, max(self.ystremes)+self.y_margin])

            # Reverse x-axis if chosen
            if self.reverse_x:
                self.ax.set_xlim([max(self.xstremes)+self.x_margin, min(self.xstremes)-self.x_margin])
                self.update_annotation(x_index=self.annotation_x_index)
            else:
                self.update_annotation(x_index=cmd.get_state()-1)
        self.fig.canvas.draw()

class plot_loader:
    def __init__(self, parent, panel, tag, row, column, cspan, width, height):
        self.parent=parent
        self.display=[BooleanVar()]
        self.display[0].set(True)
        self.file_load_limit=7
        button_width=6

        self.plot_settings={'li':dict(title='Local Index',
                                      select_title='LI Types',
                                      xlab='Residue ID',
                                      ylab='Ecut [kcal/mol]',
                                      x_colname='No',
                                      reversex=False,
                                      valids=None,
                                      index_types=['PercInd1', 'PercInd2', 'RigInd', 'RicInd'],
                                      filetypes=[("Local index files", "*local_indices.dat"), ("All files", ".*")],
                                      tool_load=self.parent.tooltips.li_loader,
                                      tool_select=self.parent.tooltips.li_select,
                                      tool_display=self.parent.tooltips.li_display),
                            'gi':dict(title='Global Index',
                                      select_title='GI Types',
                                      xlab='Ecut [kcal/mol]',
                                      ylab='Index',
                                      x_colname="Egy",
                                      reversex=True,
                                      valids=None,
                                      index_types=['CCE1', 'CCE2', 'ROP1', 'ROP2', 'MCS'],
                                      filetypes=[("Global index files", "*global_indices.dat"), ("All files", ".*")],
                                      tool_load=self.parent.tooltips.gi_loader,
                                      tool_select=self.parent.tooltips.gi_select,
                                      tool_display=self.parent.tooltips.gi_display)}

        # Make frames & Plot Loading
        self.plot_frame=widgets.canvas_frame(panel, row=row, column=column, cspan=cspan, width=width, height=height)
        self.toolbar_frame=widgets.canvas_frame(panel, row=row+1, column=column, cspan=1, width=width, height=25)
        self.open_file, _=widgets.input_field(panel, text='...', disabled=False, tooltip=self.plot_settings[tag]['tool_load'],
                command=lambda file_name:self.load(tag, file_name, mode="append"),
                row=row+1, column=column+1, fspan=3, bspan=1, width=38, filetypes=self.plot_settings[tag]["filetypes"])
        # self.open_file.config(disabledforeground='black')

        # Make plot for the first time
        self.plot=line_plot(self.parent, self.plot_frame, self.toolbar_frame,
                           self.plot_settings[tag]['index_types'],
                           self.plot_settings[tag]['x_colname'],
                           x_lab=self.plot_settings[tag]['xlab'],
                           y_lab=self.plot_settings[tag]['ylab'],
                           title=self.plot_settings[tag]['title'],
                           reverse_x=self.plot_settings[tag]['reversex'])

        # Make Index Select Menu
        self.select, _=widgets.check_menu(panel, self.plot_settings[tag]['select_title'],
                command=lambda index:self.plot.draw_lines(index, self.display),
                text=self.plot_settings[tag]['index_types'], default=1,
                valids=self.plot_settings[tag]['valids'], tooltip=self.plot_settings[tag]['tool_select'],
                row=row, column=column+cspan-3, cspan=2, width=button_width, sticky=NE)

        # Make File Display Menu
        self.display, self.display_menu=widgets.check_menu(panel, 'Files', text=[],
                valids=self.file_load_limit, colors=self.plot.line_colors,
                command=lambda _, x=len(self.display)-1: self.pass_check(x, tag),
                default=1, tooltip=self.plot_settings[tag]['tool_display'],
                row=row, column=column, cspan=2, width=button_width, sticky=NW)

    def load(self, tag, file_name, mode):
        found_data=False
        if self.plot.graph_datas==[]:
            mode="overwrite"
        if mode=="overwrite":
            self.plot.graph_datas=[]
            self.plot.transition_datas=[]

        # get data
        if os.path.isfile(file_name):
            graph_data=self.parent.rd.read_gi(file_name, tag, self.parent.hb_cut_offs)
            trans_file=file_name.replace('global_indices', 'transitions').replace('local_indices', 'transitions')
            trans_data=self.parent.rd.read_transitions(trans_file, self.parent.cna_type)

            if graph_data is None:
                errors.input_error(14, (file_name, self.plot_settings[tag]['title']))
            elif self.does_data_conform(graph_data, tag, file_name):
                found_data=True
                self.plot.add_data(graph_data, trans_data)

        # make plot
        if found_data:
            if mode=="overwrite":  # Overwrite Existing plot
                self.display=[]
                self.plot.reload(self.parent)
                self.display_menu['menu'].delete(0, 'end')

            elif mode=="append":  # Add graph_data to existing plot
                self.plot.files_to_draw.append(1)
                self.plot.remake_plot()

            # add check button to display
            self.display.append(BooleanVar())
            self.display_menu['menu'].add_checkbutton(
                label=file_name,
                onvalue=True,
                offvalue=False,
                variable=self.display[-1],
                selectcolor=self.plot.line_colors[len(self.display)-1],
                command=lambda x=len(self.display)-1: self.pass_check(x, tag))
            self.display_menu['menu'].invoke('last')

        elif mode=="overwrite":
            # set display to empty & load empty plot
            self.display=[]
            self.display_menu['menu'].delete(0, 'end')
            self.plot.reload(self.parent)

    def pass_check(self, x, tag):
        check=[c.get() for c in self.display]
        if check.count(True)>self.file_load_limit:
            if hasattr(self, 'display_menu'):
                self.display_menu["menu"].invoke(x)
            errors.attention(4, (self.file_load_limit,))
        else:
            if hasattr(self, 'display_menu'):
                colno=0
                for i, disp in enumerate(check):
                    if disp:
                        self.display_menu['menu'].entryconfig(i, selectcolor=self.plot.line_colors[colno])
                        colno+=1
            types_to_draw=zip(self.plot_settings[tag]['index_types'], [i.get() for i in self.select])
            self.plot.draw_lines(types_to_draw, self.display)

    def does_data_conform(self, data, tag, file_name):
        if tag=="gi":
            data_hb_ct_offs=sorted(list(set([round(i, 2) for i in data["Egy"]])), reverse=True)
            return_value=data_hb_ct_offs==self.parent.hb_cut_offs
            if return_value==False:
                errors.input_error(16, (file_name,))
        elif tag=="li":
            residue_count=len(self.parent.pdb_obj.res_objects)
            min_index_value=min(min([list(data[index_type]) for index_type in self.plot_settings["li"]["index_types"]]))
            return_value=residue_count==len(data["No"]) and min_index_value>=min(self.parent.hb_cut_offs)
            if return_value==False:
                errors.input_error(15, (file_name,))
        return return_value

class image_plot:
    def __init__(self, parent, image_frame, toolbar_frame, datas, x_lab, y_lab, title=''):
        # color maps
        cdict_1={'red':  ((0.0, 0.0, 0.4), (0.25, 1.0, 1.0), (0.5, 1.0, 0.8), (0.75, 0.0, 0.0), (1.0, 0.0, 0.0)),
                   'green':((0.0, 0.0, 0.0), (0.25, 0.0, 0.0), (0.5, 0.9, 0.9), (0.75, 0.0, 0.0), (1.0, 0.0, 0.0)),
                   'blue': ((0.0, 0.0, 0.0), (0.25, 0.0, 0.0), (0.5, 0.8, 1.0), (0.75, 1.0, 1.0), (1.0, 0.4, 1.0))
                  }
        from matplotlib.colors import LinearSegmentedColormap, ListedColormap
        from matplotlib import cm
        self.cmap_blue_red=LinearSegmentedColormap('BlueRed1', cdict_1)
        self.cmap_white=ListedColormap(['white'])
        self.cmap_jet=cm.get_cmap('jet_r')

        # Set figure size
        width=int(image_frame.config()["width"][-1])
        height=int(image_frame.config()["height"][-1])
        matplotlib_dpi=float(pylab.gcf().get_dpi())
        self.fig=pylab.figure(figsize=(width/matplotlib_dpi, height/matplotlib_dpi))

        # Set plot data
        self.parent=parent
        self.cur_xy_indices=[0, 0]
        self.datas=datas
        self.data=datas[0] if self.datas!=[] else None

        # Set Empty Image
        residue_count=len(self.parent.pdb_obj.res_objects)
        self.ax=self.fig.add_subplot(111)
        self.image=self.ax.imshow(np.zeros(shape=(residue_count, residue_count)), picker=5, origin="lower", cmap=self.cmap_white)
        self.image.set_clim(-6, 0)
        # Set Image Axes
        self.ax.xaxis.set_major_locator(MaxNLocator(5))
        self.ax.yaxis.set_major_locator(MaxNLocator(5))
        self.ax.xaxis.set_major_formatter(FuncFormatter(self.__resid_formatter))
        self.ax.yaxis.set_major_formatter(FuncFormatter(self.__resid_formatter))

        # Set Annotations
        self.colorbar=self.fig.colorbar(self.image)
        self.annotation=self.ax.text(-0.6,-0.1, '',
                                     horizontalalignment='left',
                                     verticalalignment='top',
                                     transform=self.ax.transAxes)
        pylab.subplots_adjust(left=0.2, right=0.7, top=0.9, bottom=0.15)

        # Set axes
        self.ax.set_xlabel(x_lab)
        self.ax.set_ylabel(y_lab)
        if self.datas==[]:
            self.ax.set_title("No %s loaded"%title)
        else:
            self.ax.set_title(title)

        # Load canvas
        canvas=FigureCanvasTkAgg(self.fig, master=image_frame)
        canvas.get_tk_widget().pack()
        canvas.mpl_connect('button_press_event', self.annotate)

        # Load toolbar
        toolbar=MyCustomToolbar(canvas, toolbar_frame, self.parent)
        toolbar.update()
        canvas._tkcanvas.pack(side=TOP, fill=BOTH, expand=1)
        self.fig.canvas.draw()

    def __resid_formatter(self, x, pos):
        'The two args are the value and tick position'
        if x==0:
            return str(self.parent.pdb_obj.res_objects[int(x)].id[1]-1)
        try:
            return self.parent.pdb_obj.resi_serial_pymol_selector[x].replace('/', '')
        except:
            return ''

    def draw_image(self, display):
        files_to_draw=[d.get() for d in display]
        index_to_draw=[i for i, d in enumerate(files_to_draw) if d]
        residue_count=len(self.parent.pdb_obj.res_objects)
        self.image.set_extent([0, residue_count, 0, residue_count])

        if len(index_to_draw)==0:
            # remove the plot
            if len(display)==0:
                self.ax.set_title("No Stability Map loaded")
            else:
                self.ax.set_title("Stability Map")
            self.data=None
            self.image.set_data(np.zeros(shape=(residue_count, residue_count)))
            self.image.set_cmap(self.cmap_white)
            self.fig.canvas.draw()

        elif len(index_to_draw) in [1, 2]:
            if len(index_to_draw)==1:
                self.ax.set_title("Stability Map")
                self.data=self.datas[index_to_draw[0]]
                percentile_data_filtered=self.data[~np.isnan(self.data)]
                lower_limit, upper_limit=np.percentile(percentile_data_filtered, 2), np.percentile(percentile_data_filtered, 98)
                self.image.set_clim(lower_limit, upper_limit)
                self.image.set_cmap(self.cmap_jet)
            elif len(index_to_draw)==2:
                self.ax.set_title("Difference stability map")
                self.data=self.datas[index_to_draw[1]]-self.datas[index_to_draw[0]]
                percentile_data_filtered=self.data[~np.isnan(self.data)]
                lower_limit, upper_limit=np.percentile(percentile_data_filtered, 2), np.percentile(percentile_data_filtered, 98)
                lim=max(abs(lower_limit), abs(upper_limit))
                self.image.set_clim(lim*-1, lim)
                self.image.set_cmap(self.cmap_blue_red)
            self.image.set_data(self.data)
            self.update_annotation(self.cur_xy_indices[0], self.cur_xy_indices[1])
            self.fig.canvas.draw()

    def annotate(self, event):
        if self.data is None:
            return
        if event.inaxes==self.ax:
            if-1<int(event.xdata)<self.data.shape[0]>int(event.ydata)>-1:
                self.cur_xy_indices[0]=int(event.xdata)
                self.cur_xy_indices[1]=int(event.ydata)
                self.update_annotation(self.cur_xy_indices[0], self.cur_xy_indices[1])
                try:
                    selection="("+self.parent.pdb_obj.resi_serial_pymol_selector[self.cur_xy_indices[0]+1]+\
                                " or "+self.parent.pdb_obj.resi_serial_pymol_selector[self.cur_xy_indices[1]+1]+\
                                ") and %s"%self.parent.pdb_obj.pdb_id
                    cmd.select("sele", selection)
                    cmd.zoom(selection)
                except:
                    pass

    def update_annotation(self, x_index, y_index):
        if self.data is None:
            return
        try:
            annotation_text="Pos: %s, %s (%.2f)"%(self.__resid_formatter(x_index+1, None), self.__resid_formatter(y_index+1, None), self.data[y_index][x_index])
        except:
            annotation_text=''
        self.annotation.set_text(annotation_text)
        self.fig.canvas.draw()


class stb_loader():

    def __init__(self, parent, panel, row, column, cspan, width, height):
        self.parent=parent
        self.display=[BooleanVar()]
        self.display[0].set(True)
        button_width=6

        # Make frames & Plot Loading
        self.image_frame=widgets.canvas_frame(panel,
                row=row, column=column, cspan=cspan, width=width, height=height)
        self.toolbar_frame=widgets.canvas_frame(panel,
                row=row+1, column=column, cspan=1, width=width, height=25)
        self.open_file, _=widgets.input_field(panel, text='...',
                command=lambda file_name:self.load(file_name, mode="append"),
                tooltip=self.parent.tooltips.stb_loader,
                row=row+1, column=column+1, fspan=3, bspan=1, width=38, disabled=False,
                filetypes=[("Stability map files", "*stability_map.dat"), ("All files", ".*")])

        # Make Plot for the first time
        file_names=[]
        datas=[]
        self.plot=image_plot(self.parent,
                             self.image_frame,
                             self.toolbar_frame,
                             datas,
                             x_lab='Residue',
                             y_lab='Residue',
                             title='Stability map')

        # Make Display
        self.display, self.display_menu=widgets.check_menu(panel, 'Files', valids=2,
                command=lambda x=None:self.plot.draw_image(self.display),
                text=file_names, tooltip=self.parent.tooltips.stb_display,
                default=1, row=row, column=column, cspan=2, width=button_width, sticky=NW)

    def load(self, file_name, mode):
        found_data=False
        if self.plot.datas==[]:
            mode="overwrite"
        if mode=="overwrite":
            self.plot.datas=[]

        # get data
        if os.path.isfile(file_name):  # TODO Bug
            data=self.parent.rd.read_stbmap(file_name)
            if data is None:
                errors.input_error(14, (file_name, "stability map"))
            elif self.does_data_conform(data, file_name):
                found_data=True
                self.plot.datas.append(data)

        # make plot
        if found_data:
            if mode=="overwrite":
                self.plot.files_to_draw=[]
                self.display=[]
                self.plot.parent=self.parent
                self.display_menu['menu'].delete(0, 'end')

            # add check button to display
            self.plot.files_to_draw.append(1)
            self.display.append(BooleanVar())
            self.display_menu['menu'].add_checkbutton(
                    label=file_name, onvalue=True, offvalue=False,
                    variable=self.display[-1],
                    command=lambda x=0:self.pass_check(x, lambda:self.plot.draw_image(self.display)))
            self.display_menu['menu'].invoke('last')
        elif mode=="overwrite":
            # set display to empty
            self.plot.files_to_draw=[]
            self.display=[]
            self.display_menu['menu'].delete(0, 'end')
            self.plot.parent=self.parent
            self.plot.draw_image(self.display)

    def pass_check(self, x, command):
        check=[c.get() for c in self.display]
        if check.count(True)>2:
            for c in self.display: c.set(False)
            self.display_menu["menu"].invoke(x)
            print 'Attention: Only 2 stability can be viewed at a time, disabled previous stability maps loaded'
            return
        else:
            command()

    def does_data_conform(self, data, file_name):
        residue_count=len(self.parent.pdb_obj.res_objects)
        return_value=data.shape[0]==residue_count
        if return_value==False:
            errors.input_error(17, (file_name,))
        return return_value

class bar_plot:
    def __init__(self, parent, plot_frame, toolbar_frame, data, x_lab, y_lab, title='',
                 sec_data=None, sec_x_lab=None, sec_y_lab=None, sec_x_tick_lab=None, pdb_seq=None, sec_binding=None):
        width=int(plot_frame.config()["width"][-1])
        height=int(plot_frame.config()["height"][-1])
        self.parent=parent
        self.data=data
        self.sec_data=sec_data
        self.type_to_draw="Type1"
        matplotlib_dpi=float(pylab.gcf().get_dpi())
        self.fig=pylab.figure(figsize=(width/matplotlib_dpi, height/matplotlib_dpi))
        self.annotation_x_index=0

        if self.sec_data is not None:
            # in case of init as empty
            if self.sec_data==[]:
                self.sec_data=[[0]*21]

            self.pdb_seq=pdb_seq
            self.sec_x_tick_lab=sec_x_tick_lab
            self.ax=pylab.subplot2grid((1, 4), (0, 0), colspan=2)
            self.sec_ax=pylab.subplot2grid((1, 4), (0, 2), colspan=2)
            pos=np.arange(21)+0.5
            self.sec_bars=self.sec_ax.bar(pos, self.sec_data[0], align='center', width=0.9, color='b', picker=5, linewidth=0)
            self.sec_ax.set_ylim(0, 100)
            self.sec_ax.set_xlim(0, 21)
            res=self.parent.pdb_obj.res_objects[self.annotation_x_index]
            chain=res.get_full_id()[2]
            resi=res.get_full_id()[3][1]
            resname=res.resname
            self.sec_ax.set_title("Substitution frequency for Res %s_%s%d"%(chain, resname, resi))
            self.sec_ax.set_xlabel(sec_x_lab)
            self.sec_ax.set_ylabel(sec_y_lab)
            pylab.xticks(pos, self.sec_x_tick_lab)
            self.sec_binding=sec_binding
        else:
            self.ax=self.fig.add_subplot(111)

        # place and format tick labels
        self.ax.xaxis.set_major_locator(MaxNLocator(5))
        self.ax.xaxis.set_major_formatter(FuncFormatter(self.__resid_formatter))

        # Set axes
        self.ax.set_xlabel(x_lab)
        self.ax.set_ylabel(y_lab)

        residue_count=len(self.parent.pdb_obj.res_objects)
        self.ax.set_xlim(0, residue_count)
        self.ax.set_ylim(0, 100+(100*0.04))
        self.bars=self.ax.bar(range(residue_count), [0]*residue_count, width=0.8, color='b', linewidth=0, picker=5)
        self.ax.set_title("No file loaded")
        self.fig.tight_layout()

        # canvas
        canvas=FigureCanvasTkAgg(self.fig, master=plot_frame)
        canvas.get_tk_widget().pack(expand=1, fill=BOTH, side=TOP)
        canvas.mpl_connect('button_press_event', self.annotate)
        self.annotation=self.ax.text(-0.1,-0.125, "", horizontalalignment='left', verticalalignment='top', transform=self.ax.transAxes)

        self.fig.canvas.draw()

        # toolbar
        toolbar=MyCustomToolbar(canvas, toolbar_frame, self.parent)
        toolbar.update()
        canvas._tkcanvas.pack(side=TOP, fill=BOTH, expand=1)

    def __resid_formatter(self, x, pos):
        'The two args are the value and tick position'
        if x==0:
            return str(self.parent.pdb_obj.res_objects[int(x)].id[1]-1)
        try:
            return self.parent.pdb_obj.resi_serial_pymol_selector[x].replace('/', '')
        except:
            return ''

    def reload(self, parent, data, sec_data=None, title=None):
        self.parent=parent
        self.data=data
        self.sec_data=sec_data

        for rect in self.bars:
            rect.remove()
        if self.data!=[]:
            try:
                self.type_to_draw=self.data.dtype.names[4]
            except:
                self.type_to_draw="Type1"
            self.annotation_x_index=0
            self.bars=self.ax.bar(range(len(self.data[self.type_to_draw])), self.data[self.type_to_draw], width=0.8, color='b', picker=5, linewidth=0)
            self.ax.set_xlim(0, len(self.data[self.type_to_draw]))
            self.update(self.type_to_draw, title)
        else:
            residue_count=len(self.parent.pdb_obj.res_objects)
            self.ax.set_xlim(0, residue_count)
            self.ax.set_ylim(0, 100+(100*0.04))
            self.bars=self.ax.bar(range(residue_count), [0]*residue_count, width=0.8, color='b', linewidth=0)
            self.ax.set_title("No file loaded")

        if self.sec_data==[]:
            for rect in self.sec_bars:
                rect.remove()
            self.sec_ax.set_xlim(0, 21)
            self.sec_ax.set_ylim(0, 100+(100*0.04))
            self.sec_bars=self.sec_ax.bar(range(21), [0]*21, width=0.8, color='b', linewidth=0)
            self.sec_ax.set_title("No file loaded")

        elif self.sec_data is not None:
            for rect in self.sec_bars:
                rect.remove()
            self.annotation_x_index=0
            pos=np.arange(len(self.sec_data[0]))+0.5
            self.sec_bars=self.sec_ax.bar(pos, self.sec_data[self.annotation_x_index], align='center', width=0.9, color='b', picker=5, linewidth=0)
            self.sec_update(self.annotation_x_index)

        self.fig.canvas.draw()

    def annotate(self, event):
        if self.data==[]:
            return

        if event.inaxes==self.ax:
            event.xdata=int(event.xdata)
            annotation_x_index=event.xdata
            self.update_annotation(annotation_x_index)
            try:
                selection=self.parent.pdb_obj.resi_serial_pymol_selector[annotation_x_index+1]+" and "+self.parent.pdb_obj.pdb_id
                zoom_selction="%s around 5"%selection
                cmd.select("sele", selection)
                cmd.zoom(zoom_selction)
                if self.sec_data is not None:
                    self.sec_update(annotation_x_index)
            except:
                pass

        if self.sec_data is None:
            return
        if event.inaxes==self.sec_ax:
            event.xdata=int(event.xdata)
            if self.sec_ax.get_navigate_mode() is not None:
                # return if navigation mode is not None This is needed otherwise the button release is not captured
                # because of attention window from the mutate and the plot keep on zooming/panning
                print "Stop zoom/pan mode to mutate."
                return
            if event.xdata<0 or event.xdata>=len(self.sec_x_tick_lab):
                return
            # mutate
            res=self.parent.pdb_obj.res_objects[self.annotation_x_index]
            chain=res.get_full_id()[2]
            resi=res.get_full_id()[3][1]
            src_resname=res.resname
            target_resname=methods.get_three_letter_resi_codes(self.sec_x_tick_lab[int(event.xdata)])
            if target_resname=="XXX":
                return
            if src_resname==target_resname:
                print "You can't mutate a residue to itself, choose another substitution."
                return
            target_frequency=self.sec_data[self.annotation_x_index][int(event.xdata)]
            methods.mutate(self.parent, self, chain, resi, src_resname, target_resname, target_frequency)


    def update_annotation(self, x_index):
        if self.data!=[]:
            if x_index<len(self.data[self.type_to_draw]) and x_index>0:
                annotation_text="Pos: %s, %.2f"%(self.__resid_formatter(x_index+1, None), self.data[self.type_to_draw][x_index])
                self.annotation.set_text(annotation_text)
                self.bars[self.annotation_x_index].set_color('blue')
                self.bars[x_index].set_color('red')
                self.fig.canvas.draw()
                self.annotation_x_index=x_index

    def update(self, unf_nuc_type, title=None):
        self.type_to_draw=unf_nuc_type
        for i, rect in enumerate(self.bars):
            rect.set_height(self.data[self.type_to_draw][i])
        if max(self.data[self.type_to_draw])==0:
            self.ax.set_ylim(0, 100+(100*0.04))
        else:
            margin_y=(max(self.data[self.type_to_draw])-min(self.data[self.type_to_draw]))*0.04
            self.ax.set_ylim(min(self.data[self.type_to_draw]), max(self.data[self.type_to_draw])+margin_y)
            if title is None:
                self.ax.set_title("Weak spot "+self.type_to_draw)
            else:
                self.ax.set_title(title)
        self.fig.canvas.draw()
        self.update_annotation(self.annotation_x_index)

    def sec_update(self, x_index):
        new_data=self.sec_data[x_index]
        for i, rect in enumerate(self.sec_bars):
            rect.set_height(new_data[i])
        res=self.parent.pdb_obj.res_objects[self.annotation_x_index]
        chain=res.get_full_id()[2]
        resi=res.get_full_id()[3][1]
        resname=res.resname
        self.sec_ax.set_title("Substitution frequency for Res %s_%s%d"%(chain, resname, resi))
        self.fig.canvas.draw()

class unf_loader:

    def __init__(self, parent, panel, row, column, cspan, width, height):
        self.parent=parent
        self.display=[BooleanVar()]
        self.display[0].set(True)
        button_width=6

        # make plot axes
        self.plot_frame=widgets.canvas_frame(panel,
                row=row, column=column, cspan=cspan, width=width, height=height)
        self.toolbar_frame=widgets.canvas_frame(panel,
                row=row+1, column=column, cspan=1, width=width, height=25)

        self.open_file, _=widgets.input_field(panel, text='...',
                command=lambda file_name:self.load(file_name),
                tooltip=self.parent.tooltips.unf_loader,
                row=row+1, column=column+1, fspan=3, bspan=1, width=38, disabled=False,
                filetypes=[("Unfolding nuclei files", "*unfolding_nuclei.dat"), ("All files", ".*")])

        self.open_file.config(disabledforeground='black')

        # Make Plot for the example file
        self.datas=[]
        self.files=[]
        unf_nuc_types=['None']

        # Make Selection
        self.select, self.select_menu=widgets.click_menu(panel, text=unf_nuc_types, default=0,
                commands=(lambda i:self.plot.update(i) for i in unf_nuc_types),
                tooltip=self.parent.tooltips.unf_select,
                row=row, column=column+cspan-3, cspan=2, width=button_width, sticky=NE)

        # Make display
        self.display, self.display_menu=widgets.check_menu(panel, 'Files',
                command=lambda :self.load(self.open_file.get()), valids=1,
                text=self.files, tooltip=self.parent.tooltips.unf_display,
                default=1, row=row, column=column, cspan=2, width=button_width, sticky=NW)

        self.plot=bar_plot(self.parent, self.plot_frame, self.toolbar_frame,
                          self.datas, x_lab='Residue', y_lab='Frequency (%)', title='Weak spots Type 1')

    def pass_check(self, x):
        check=[c.get() for c in self.display]
        if check.count(True)>1:
            for c in self.display: c.set(False)
            self.display_menu["menu"].invoke(x)
            print 'Attention: Only one unfolding nuclei file can be viewed at a time, disabled previous unfolding nuclei file loaded'
            return
        elif check.count(True)!=1:
            self.plot.reload(self.parent, [])
            unf_nuc_types=[]
        else:
            self.plot.reload(self.parent, self.datas[x])
            if self.datas[x]!=[]:
                unf_nuc_types=self.plot.data.dtype.names[4:]
            else:
                unf_nuc_types=[]

        def pass_command(cmd, label):
            self.select.set(label)
            cmd

        self.select_menu['menu'].delete(0, 'end')
        for unf_type in unf_nuc_types:
            self.select_menu['menu'].add_command(
                    label=unf_type,
                    command=lambda x=unf_type:pass_command(self.plot.update(x), x))
        if self.datas[x]!=[]:
            self.select_menu['menu'].invoke(0)

    def load(self, file_name, mode='append'):

        found_data=False
        if self.datas==[[]]:
            mode='overwrite'
        if mode=='overwrite':
            self.datas=[]

        # Get data
        if os.path.isfile(file_name):
            data=self.parent.rd.read_unf_nuc(file_name, cna_method=self.parent.cna_type, pdb_obj=self.parent.pdb_obj)
            if data is None:
                self.datas.append([])
            else:
                self.datas.append(data)
                found_data=True
        else:
            self.datas.append([])

        if found_data:
            if mode=='overwrite':
                self.files=[]
                self.display=[]
                self.plot.parent=self.parent
                self.display_menu['menu'].delete(0, 'end')

            self.files.append(file_name)
            self.display.append(BooleanVar())
            self.display_menu['menu'].add_checkbutton(
                    label=file_name, onvalue=True, offvalue=False,
                    variable=self.display[-1],
                    command=lambda x=len(self.display)-1:self.pass_check(x))
            self.display_menu['menu'].invoke('last')

        elif mode=='overwrite':
            self.files=[]
            self.display=[]
            self.plot.parent=self.parent
            self.display_menu['menu'].delete(0, 'end')

        self.plot.reload(self.parent, self.datas[0])

class msa_loader:
    def __init__(self, parent, panel, row, column, cspan, width, height, example_file, example_seq_name, example_chain_id):
        self.parent=parent
        self.display=[BooleanVar()]
        self.display[0].set(True)
        button_width=6

        # make plot axes
        MSA_formats=[("MSA Files", "*.clustal"),
                       ("MSA Files", "*.pir"),
                       ("MSA Files", "*.sth"),
                       ("MSA Files", "*.fasta"),
                       ("MSA Files", "*.emboss"),
                       ("All files", "*")]
        self.plot_frame=widgets.canvas_frame(panel,
                row=row, column=column, cspan=cspan, width=width, height=height)
        self.toolbar_frame=widgets.canvas_frame(panel,
                row=row+1, column=column, cspan=1, width=width, height=25)
        self.open_file, _=widgets.input_field(panel, text='...', default=example_file,
                tooltip=self.parent.tooltips.msa_loader,
                row=row+1, column=column+1, fspan=2, bspan=1, width=48, disabled=False,
                filetypes=MSA_formats)

        widgets.label_field(panel, "Sequence name:",
                row=row+1, column=column+4, width=14)
        self.basename=widgets.entry_field(panel, example_seq_name,
                tooltip=self.parent.tooltips.sequence_name,
                row=row+1, column=column+6, width=15, sticky=W)
        widgets.label_field(panel, "Chain ID:",
                row=row+1, column=column+7, width=8)
        self.chain_id=widgets.entry_field(panel, example_chain_id,
                tooltip=self.parent.tooltips.chain_id_name,
                row=row+1, column=column+8, width=2, sticky=W)
        widgets.call_button(panel, tooltip=self.parent.tooltips.open_msa_file,
                command=lambda :self.load(self.open_file.get(), self.basename.get(), self.chain_id.get()), text='load',
                row=row+1, width=5, column=column+9, sticky=W)

        # Make Plot for the example file if provided
        aa_list='-ACDEFGHIKLMNPQRSTVWY'
        self.files=[]
        if os.path.isfile(example_file) and example_seq_name!='' and example_chain_id!=[]:
            aa_frequency_datas, base_seq_conservations=self.parent.rd.read_msa(msa_file=example_file,
                                                                               pdb_obj=parent.pdb_obj,
                                                                               chain_id=example_chain_id,
                                                                               base_seq_name=example_seq_name)
            self.conservation_datas=[{'Type1': base_seq_conservations}]
            self.aa_frequency_datas=[aa_frequency_datas]
        else:
            self.conservation_datas=[[]]
            self.aa_frequency_datas=[[]]

        self.plot=bar_plot(self.parent, self.plot_frame, self.toolbar_frame,
                           self.conservation_datas[0],
                           x_lab='Residue', y_lab='Conservation (%)', title='Residue conservation',
                           sec_data=self.aa_frequency_datas[0],
                           sec_x_lab='Amino acid', sec_y_lab='Conservation (%)', sec_x_tick_lab=list(aa_list), sec_binding="mutate")
        # Make display
        self.display, self.display_menu=widgets.check_menu(panel, 'Files',
                command=lambda x=None: self.pass_check(0), valids=1,
                text=self.files, tooltip=self.parent.tooltips.msa_display,
                default=1, row=row, column=column, cspan=2, width=button_width, sticky=NW)

    def pass_check(self, x):
        check=[c.get() for c in self.display]
        if check.count(True)>1:
            for c in self.display: c.set(False)
            self.display_menu["menu"].invoke(x)
            print 'Attention: Only one MSA file can be viewed at a time, disabled previous MSA file loaded'
        elif check.count(True)==1:
            self.plot.reload(self.parent, self.conservation_datas[x], sec_data=self.aa_frequency_datas[x], title="Residue conservation")
        else:
            self.plot.reload(self.parent, [], sec_data=[])

    def load(self, file_name, seq_name, chain_name, mode="append"):

        found_data=False
        if self.conservation_datas==[[]]:
            mode='overwrite'
        if mode=='overwrite':
            self.conservation_datas=[]
            self.aa_frequency_datas=[]

        # Get data
        if os.path.isfile(file_name):
            aa_frequency_data, base_seq_conservations=self.parent.rd.read_msa(msa_file=file_name,
                                                                               pdb_obj=self.parent.pdb_obj,
                                                                               chain_id=chain_name,
                                                                               base_seq_name=seq_name)
            # print aa_frequency_datas, base_seq_conservations
            if aa_frequency_data is None:
                self.conservation_datas.append([])
                self.aa_frequency_datas.append([])
            else:
                self.conservation_datas.append({"Type1":base_seq_conservations})
                self.aa_frequency_datas.append(aa_frequency_data)
                found_data=True
        else:
            self.conservation_datas.append([])
            self.aa_frequency_datas.append([])


        if found_data:
            if mode=='overwrite':
                self.files=[]
                self.display=[]
                self.plot.parent=self.parent
                self.display_menu['menu'].delete(0, 'end')

            self.files.append(file_name)
            self.display.append(BooleanVar())
            self.display_menu['menu'].add_checkbutton(
                    label=file_name, onvalue=True, offvalue=False,
                    variable=self.display[-1],
                    command=lambda x=len(self.display)-1:self.pass_check(x))
            self.display_menu['menu'].invoke('last')

        elif mode=='overwrite':
            self.files=[]
            self.display=[]
            self.plot.parent=self.parent
            self.display_menu['menu'].delete(0, 'end')
        self.plot.reload(self.parent, self.conservation_datas[0], self.aa_frequency_datas[0], title="Residue conservation")
