"""
    Created Mar 2014
    This file is part of VisualCNA
    Copyright (2014) Prakash Chandra Rathi and Daniel Mulnaes

    VisualCNA is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    VisualCNA is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
#####################################################################
# -*- coding: utf-8 -*-
#
# Authors: Prakash Chandra Rathi and Daniel Mulnaes
# Heinrich Heine University, Duesseldorf
# Institute for Pharmaceutical and Medicinal Chemistry
# Universitaetsstr. 1 40225 Duesseldorf
# Germany
#####################################################################

# Standard python libraries
from Tkinter import *

# External Modules
from pymol import cmd
from tkintertable.Tables import TableCanvas, ColumnHeader, RowHeader
from tkintertable import Table_images

class MyTableCanvas(TableCanvas):
    """A tkinter class for providing table functionality"""
    def do_bindings(self):
        """Bind keys and mouse clicks, this can be overriden"""
        self.bind("<Button-1>", self.handle_left_click)
        self.bind("<Double-Button-1>", self.handle_double_click)
        # self.bind("<Control-Button-1>", self.handle_left_ctrl_click)
        # self.bind("<Shift-Button-1>", self.handle_left_shift_click)

        # self.bind("<ButtonRelease-1>", self.handle_left_release)
        if self.ostyp=='mac':
            # For mac we bind Shift, left-click to right click
            self.bind("<Button-2>", self.handle_right_click)
            self.bind('<Shift-Button-1>', self.handle_right_click)
        else:
            self.bind("<Button-3>", self.handle_right_click)

        # self.bind('<B1-Motion>', self.handle_mouse_drag)
        # self.bind('<Motion>', self.handle_motion)

        # self.bind_all("<Control-x>", self.delete_Row)
        # self.bind_all("<Control-n>", self.add_Row)
        # self.bind_all("<Delete>", self.clearData)
        # self.bind_all("<Control-v>", self.paste)

        # if not hasattr(self,'parentapp'):
        #    self.parentapp = self.parentframe

        # TODO: check if key bindings to arrow keys can be restored
        # self.parentframe.master.bind_all("<Right>", self.handle_arrow_keys)
        # self.parentframe.master.bind_all("<Left>", self.handle_arrow_keys)
        # self.parentframe.master.bind_all("<Up>", self.handle_arrow_keys)
        # self.parentframe.master.bind_all("<Down>", self.handle_arrow_keys)
        # self.parentframe.master.bind_all("<KP_8>", self.handle_arrow_keys)
        # self.parentframe.master.bind_all("<Return>", self.handle_arrow_keys)
        # self.parentframe.master.bind_all("<Tab>", self.handle_arrow_keys)
        if 'windows' in self.platform:
            self.bind("<MouseWheel>", self.mouse_wheel)
        self.bind('<Button-4>', self.mouse_wheel)
        self.bind('<Button-5>', self.mouse_wheel)
        return

    def mouse_wheel(self, event):
        """Handle mouse wheel scroll for windows"""
        if self.rows<16:
            return
        if event.num==5 or event.delta==-120:
            event.widget.yview_scroll(1, UNITS)
            self.tablerowheader.yview_scroll(1, UNITS)
        if event.num==4 or event.delta==120:
            event.widget.yview_scroll(-1, UNITS)
            self.tablerowheader.yview_scroll(-1, UNITS)
        return

    def handle_double_click(self, event):
        """Do double click stuff. Selected row/cols will already have
           been set with single click binding"""
        row=self.get_row_clicked(event)
        # col=self.get_col_clicked(event)
        absrow=self.get_AbsoluteRow(row)
        model=self.getModel()
        # cellvalue=model.getCellRecord(absrow, col)
        atom_1=model.getCellRecord(absrow, 1)
        atom_2=model.getCellRecord(absrow, 2)
        zoom_selection="id %s or id %s"%(atom_1, atom_2)
        cmd.select("sele", zoom_selection)
        cmd.zoom(zoom_selection)
        return

    def popupMenu(self, event, rows=None, cols=None, outside=None):
        """Add left and right click behaviour for canvas, should not have to override
            this function, it will take its values from defined dicts in constructor"""
        if outside==None:
            row=self.get_row_clicked(event)
            col=self.get_col_clicked(event)
            # coltype=self.model.getColumnType(col)
        popupmenu=Menu(self, tearoff=0)
        def popupFocusOut(event):
            popupmenu.unpost()

        if outside==1:
            # if outside table, just show general items
            return popupmenu
            popupmenu.add_command(label="Resize Columns", command=self.autoResizeColumns)
            popupmenu.add_command(label="Filter Recs", command=self.showFilteringBar)
            popupmenu.add_command(label="Save Table", command=self.saveTable)
            popupmenu.add_command(label="Export Table", command=self.exportTable)
            popupmenu.add_command(label="Preferences", command=self.showtablePrefs)
        else:
            def add_commands(fieldtype):
                """Add commands to popup menu for col type"""
                # add column actions for this table type defined in self.columnactions
                functions=self.columnactions[fieldtype]
                for f in functions.keys():
                    func=getattr(self, functions[f])
                    popupmenu.add_command(label=f, command=lambda : func(row, col))
                return

            def add_defaultcommands():
                """now add general actions for all cells"""
                #===============================================================
                # main = ["Set Fill Color","Set Text Color","Copy", "Paste", "Fill Down","Fill Right", "Clear Data",
                #          "Add Row" , "Delete Row", "Select All", "Resize Columns", "Plot Selected",
                #          "Plot Options", "Preferences"]
                #===============================================================
                main=["Set Fill Color", "Set Text Color"]
                # utils=["View Record", "Formulae->Value", "Export Table"]

                defaultactions={"Set Fill Color" : lambda : self.setcellColor(rows, cols, key='bg'),
                                "Set Text Color" : lambda : self.setcellColor(rows, cols, key='fg'),
                                "Copy" : lambda : self.copyCell(rows, cols),
                                "Paste" : lambda : self.pasteCell(rows, cols),
                                "Fill Down" : lambda : self.fill_down(rows, cols),
                                "Fill Right" : lambda : self.fill_across(cols, rows),
                                "Add Row" : lambda : self.add_Row(),
                                "Delete Row" : lambda : self.delete_Row(),
                                "View Record" : lambda : self.getRecordInfo(row),
                                "Clear Data" : lambda : self.delete_Cells(rows, cols),
                                "Select All" : self.select_All,
                                "Resize Columns" : self.autoResizeColumns,
                                "Plot Selected" : self.plot_Selected,
                                "Plot Options" : self.plotSetup,
                                "Export Table" : self.exportTable,
                                "Preferences" : self.showtablePrefs,
                                "Formulae->Value" : lambda : self.convertFormulae(rows, cols)}


                for action in main:
                    if action=='Fill Down' and (rows==None or len(rows)<=1):
                        continue
                    if action=='Fill Right' and (cols==None or len(cols)<=1):
                        continue
                    else:
                        popupmenu.add_command(label=action, command=defaultactions[action])
                # popupmenu.add_separator()
                # utilsmenu = Menu(popupmenu, tearoff = 0)
                # popupmenu.add_cascade(label="Utils",menu=utilsmenu)
                # for action in utils:
                #    utilsmenu.add_command(label=action, command=defaultactions[action])
                return

            # if self.columnactions.has_key(coltype):
            #    add_commands(coltype)
            add_defaultcommands()

        popupmenu.bind("<FocusOut>", popupFocusOut)
        popupmenu.focus_set()
        popupmenu.post(event.x_root, event.y_root)
        return popupmenu


    def draw_selected_rect(self, row, col, color=None):
        """User has clicked to select a cell"""
        if col>=self.cols:
            return
        self.delete('currentrect')
        # bg=self.selectedcolor
        bg=None
        if color==None:
            color='gray25'
        w=3
        x1, y1, x2, y2=self.getCellCoords(row, col)
        self.create_rectangle(x1+w/2, y1+w/2, x2-w/2, y2-w/2,
                              fill=bg,
                              outline=color,
                              width=w,
                              stipple='gray50',
                              tag='currentrect')
        # self.lower('currentrect')
        # raise text above all
        self.lift('celltext'+str(col)+'_'+str(row))
        return

    def drawNavFrame(self):
        """Draw the frame for selecting pages when paging is on"""
        self.navFrame=Frame(self.parentframe)
        self.navFrame.grid(row=4, column=0, columnspan=2, sticky='news', padx=1, pady=1, ipady=1)
        pagingbuttons={ 'start' : self.first_Page, 'prev' : self.prev_Page,
                          'next' : self.next_Page, 'end' : self.last_Page}
        images={ 'start' : Table_images.start(), 'prev' : Table_images.prev(),
                   'next' : Table_images.next(), 'end' : Table_images.end()}
        skeys=['start', 'prev', 'next', 'end']
        for i in skeys:
            b=Button(self.navFrame, text=i, command=pagingbuttons[i],
                        relief=GROOVE,
                        image=images[i])
            b.image=images[i]
            b.pack(side=LEFT, ipadx=1, ipady=1)
        Label(self.navFrame, text='Page '+str(self.currentpage+1)+' of '+str(self.numpages), fg='white',
                  bg='#3366CC', relief=SUNKEN).pack(side=LEFT, ipadx=2, ipady=2, padx=4)
        # Label(self.navFrame,text='Goto Record:').pack(side=LEFT,padx=3)
        # self.gotorecordvar = StringVar()
        # Entry(self.navFrame,textvariable=self.gotorecordvar,
        #          width=8,bg='white').pack(side=LEFT,ipady=3,padx=2)
        Label(self.navFrame, text=str(self.rows)+' records').pack(side=LEFT, padx=3)
        # Button(self.navFrame,text='Normal View',command=self.paging_Off,
        #           bg='#99CCCC',relief=GROOVE).pack(side=LEFT,padx=3)
        return


    def findValue(self, searchstring=None, direction="start"):
        """Return the row/col for the input value"""
        if searchstring==None:
            searchstring=tkSimpleDialog.askstring("Search table.",
                                               "Enter search value",
                                               parent=self.parentframe)
        found=0
        if direction not in ["forward", "backward"] or not hasattr(self, 'foundlist'):
            self.foundlist=[]
        if self.model!=None:
            if direction=="backward" and len(self.foundlist)>1:
                found=1
                row, col=self.foundlist[-2]
                # remove last found from foundlist
                self.foundlist.pop()
                self.go_to_found_cell(row, col)
                return row, col
            elif direction in ["start", "forward"]:
                for row in range(self.rows):
                    for col in range(self.cols):
                        text=str(self.model.getValueAt(row, col))
                        if text=='' or text==None:
                            continue
                        cell=row, col
                        if direction=="forward" and cell in self.foundlist:
                            continue
                        if text.lower().find(searchstring.lower())!=-1:
                            found=1
                            # add row/col to foundlist
                            self.foundlist.append(cell)
                            self.go_to_found_cell(row, col)
                            return row, col
        if found==0:
            self.delete('searchrect')
            return None

    def go_to_found_cell(self, row, col):
        # need to scroll to centre the cell here..
        x, y, new_page, new_row=self.getCanvasPos(row, col)
        # move to new_page
        # added paging support
        if new_page!=self.currentpage:
            self.currentpage=new_page
            self.redrawTable()
        self.xview('moveto', x)
        self.yview('moveto', y)
        self.tablecolheader.xview('moveto', x)
        self.tablerowheader.yview('moveto', y)
        # highlight cell
        self.delete('searchrect')
        self.draw_rect(new_row, col, color='red', tag='searchrect', delete=0)
        self.lift('searchrect')
        self.lift('celltext'+str(col)+'_'+str(new_row))


    def getCanvasPos(self, row, col):
        """Get the cell x-y coords as a fraction of canvas size"""
        paging=self.paging
        if self.rows==0:
            return None, None
        if paging==1:
            page=row/self.rowsperpage
            new_row=row%self.rowsperpage
            x1, y1, x2, y2=self.getCellCoords(new_row, col)
            cx=float(x1)/self.tablewidth
            # Account for 10 pixels added in the bottom of the table
            cy=float(y1)/((self.rowsperpage*self.rowheight)+10)
            return cx, cy, page, new_row
        else:
            page=0
            x1, y1, x2, y2=self.getCellCoords(row, col)
            cx=float(x1)/self.tablewidth
            cy=float(y1)/((self.rows*self.rowheight)+10)
            return cx, cy, page, row

    def createTableFrame(self, callback=None):
        """Adds column header and scrollbars and combines them with
           the current table adding all to the master frame provided in constructor.
           Table is then redrawn."""

        # Add the table and header to the frame
        self.Yscrollbar=Scrollbar(self.parentframe, orient=VERTICAL, command=self.set_yviews)
        self.Yscrollbar.grid(row=1, column=2, rowspan=1, sticky='news', pady=0, ipady=0)
        # self.Xscrollbar=Scrollbar(self.parentframe, orient=HORIZONTAL, command=self.set_xviews)
        # self.Xscrollbar.grid(row=2, column=1, columnspan=1, sticky='news')
        # self['xscrollcommand']=self.Xscrollbar.set
        self['yscrollcommand']=self.Yscrollbar.set
        # self.tablecolheader['xscrollcommand']=self.Xscrollbar.set
        self.tablerowheader['yscrollcommand']=self.Yscrollbar.set
        self.parentframe.rowconfigure(1, weight=1)
        self.parentframe.columnconfigure(1, weight=1)

        self.tablecolheader.grid(row=0, column=1, rowspan=1, sticky='news', pady=0, ipady=0)
        self.tablerowheader.grid(row=1, column=0, rowspan=1, sticky='news', pady=0, ipady=0)
        self.grid(row=1, column=1, rowspan=1, sticky='news', pady=0, ipady=0)
        if self.model.getRowCount()<500:
            self.adjustColumnWidths()
        self.redrawTable(callback=callback)
        # self.parentframe.bind("<Configure>", self.resizeTable)
        self.tablecolheader.xview("moveto", 0)
        self.xview("moveto", 0)
        return


class MyColumnHeader(ColumnHeader):
    """Class that takes it's size and rendering from a parent table
        and column names from the table model."""

    def __init__(self, parent=None, table=None):
        Canvas.__init__(self, parent, bg='gray25', width=500, height=20)
        self.thefont='Arial 14'
        if table!=None:
            self.table=table
            self.height=20
            self.model=self.table.getModel()
            self.config(width=self.table.width)
            # self.colnames = self.model.columnNames
            self.columnlabels=self.model.columnlabels
            # self.bind('<Button-1>',self.handle_left_click)
            # self.bind("<ButtonRelease-1>", self.handle_left_release)
            # self.bind('<B1-Motion>', self.handle_mouse_drag)
            # self.bind('<Motion>', self.handle_mouse_move)
            # self.bind('<Shift-Button-1>', self.handle_left_shift_click)
            if self.table.ostyp=='mac':
                # For mac we bind Shift, left-click to right click
                self.bind("<Button-2>", self.handle_right_click)
                self.bind('<Shift-Button-1>', self.handle_right_click)
            else:
                self.bind("<Button-3>", self.handle_right_click)
            self.thefont=self.table.thefont
        return

    def popupMenu(self, event):
        """Add left and right click behaviour for column header"""
        colname=self.model.columnNames[self.table.currentcol]
        collabel=self.model.columnlabels[colname]
        currcol=self.table.currentcol
        popupmenu=Menu(self, tearoff=0)
        def popupFocusOut(event):
            popupmenu.unpost()
        # popupmenu.add_command(label="Rename Column", command=self.relabel_Column)
        popupmenu.add_command(label="Sort by "+collabel, command=lambda : self.table.sortTable(currcol))
        popupmenu.add_command(label="Sort by "+collabel+' (descending)', command=lambda : self.table.sortTable(currcol, reverse=1))
        # popupmenu.add_command(label="Delete This Column", command=self.table.delete_Column)
        # popupmenu.add_command(label="Add New Column", command=self.table.add_Column)

        popupmenu.bind("<FocusOut>", popupFocusOut)
        # self.bind("<Button-3>", popupFocusOut)
        popupmenu.focus_set()
        popupmenu.post(event.x_root, event.y_root)
        return popupmenu

class MyRowHeader(RowHeader):
    """Class that displays the row headings on the table
       takes it's size and rendering from the parent table
       This also handles row/record selection as opposed to cell
       selection"""
    def __init__(self, parent=None, table=None):
        Canvas.__init__(self, parent, bg='gray75', width=40, height=None)

        if table!=None:
            self.table=table
            self.width=40
            self.x_start=40
            self.inset=1
            # self.config(width=self.width)
            self.config(height=self.table.height)
            self.startrow=self.endrow=None
            self.model=self.table.getModel()
            self.bind('<Button-1>', self.handle_left_click)
            self.bind("<ButtonRelease-1>", self.handle_left_release)
            self.bind("<Control-Button-1>", self.handle_left_ctrl_click)
            self.bind('<Button-3>', self.handle_right_click)
            self.bind('<B1-Motion>', self.handle_mouse_drag)
            # self.bind('<Shift-Button-1>', self.handle_left_shift_click)

        return

    def redraw(self, paging=0):
        """Redraw row header"""
        if paging==1 and self.table.rows>self.table.rowsperpage:
            self.height=self.table.rowheight*self.table.rowsperpage+10
        else:
            self.height=self.table.rowheight*self.table.rows+10

        self.configure(scrollregion=(0, 0, self.width, self.height))
        self.delete('rowheader', 'text')
        self.delete('rect')

        w=1
        x_start=self.x_start
        y_start=self.table.y_start
        h=self.table.rowheight
        rowpos=0

        # Delete all previous checkboxes
        for box in self.checkbox_mapping:
            self.delete(self.checkbox_mapping[box]['window'])

        for row in self.table.rowrange:
            x1, y1, x2, y2=self.table.getCellCoords(rowpos, 0)
            self.create_rectangle(0, y1, x_start-w, y2,
                                      fill='gray75',
                                      outline='white',
                                      width=w,
                                      tag='rowheader')

            try:
                constraint_row=rowpos if self.table.currentpage is None else self.table.currentpage*self.table.rowsperpage+rowpos
                constraint_cid=(int(self.model.getValueAt(constraint_row, 1)), int(self.model.getValueAt(constraint_row, 2)))
                self.create_check_box(20, y1+h/2, fill='white', cid=constraint_cid, anchor='center')
            except:
                pass

            rowpos+=1
        return

    def update_net_obj(self, button):
        label=self.net_parent.net_obj.id_to_label[button.cid]
        zoom_selection="id %s or id %s"%(button.cid[0], button.cid[1])
        cmd.select("sele", zoom_selection)
        if button.var.get():
            self.net_parent.net_obj.create_constraints([label], table_update=False)
            cmd.zoom(zoom_selection)
        else:
            cmd.zoom(zoom_selection)
            self.net_parent.net_obj.delete_constraints([label], table_update=False)

    def create_check_box(self, x, y, fill, cid, anchor):
        check_var=BooleanVar()
        if self.net_parent.net_obj.constraints[cid]['status']!=-1:
            check_var.set(True)
        else:
            check_var.set(False)

        check_button=Checkbutton(self, variable=check_var, command=lambda i=id:self.update_net_obj(check_button), onvalue=True, offvalue=False,
                                   borderwidth=0, padx=0, pady=0, bg="grey", highlightthickness=0, highlightbackground='grey')
        check_button.var=check_var
        check_button.cid=cid
        window=self.create_window(x, y, anchor=anchor, window=check_button)
        self.checkbox_mapping[cid]={'button':check_button, 'window':window, 'coords':(x, y)}
