"""
    Created Mar 2014
    This file is part of VisualCNA
    Copyright (2014) Prakash Chandra Rathi and Daniel Mulnaes

    VisualCNA is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    VisualCNA is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
#####################################################################
# -*- coding: utf-8 -*-
#
# Authors: Daniel Mulnaes and Prakash Chandra Rathi
# Heinrich Heine University, Duesseldorf
# Institute for Pharmaceutical and Medicinal Chemistry
# Universitaetsstr. 1 40225 Duesseldorf
# Germany
#####################################################################


# Standard python libraries
import os, tkMessageBox, tkFileDialog
from Tkinter import *
# from PIL import ImageTk, Image

# External modules
import Pmw
from pymol.cmd import *

# VisualCNA Modules
import methods

def image_frame(parent, image, row=0, column=0, cspan=1, rspan=1, width=20, height=20):
    frame=Canvas(parent, width=width, height=height)
    frame.grid(row=row, column=column, rowspan=rspan, columnspan=cspan)
    frame.img=PhotoImage(file=image, width=width, height=height)
    label=Label(frame, image=frame.img, width=width, height=height)
    label.grid(row=row, column=column, rowspan=rspan, columnspan=cspan)

def canvas_frame(parent, row=0, column=0, cspan=1, rspan=1, width=20, height=20, bd=0, tooltip=None):
    frame=Canvas(parent, width=width, height=height, bd=bd)
    frame.grid(row=row, column=column, rowspan=rspan, columnspan=cspan, sticky=NW)
    if tooltip is not None:
        Pmw.Balloon(parent).bind(frame, balloonHelp=tooltip)
    return frame

def check_button(parent, text='Checkbutton', on='ON', off='OFF', command=lambda x:True,
                 selected=False, disabled=False, sticky=N+E+W+S, tooltip=None,
                 row=0, column=0, cspan=1, rspan=1, width=20, height=1):
    if type(on) is bool and type(off) is bool:
        var=BooleanVar()
    elif type(on) is str and type(off) is str:
        var=StringVar()

    def pass_command():
        command(var.get())

    check_button=Checkbutton(parent, text=text, variable=var, command=pass_command, onvalue=on, offvalue=off)
    check_button.grid(row=row, column=column, rowspan=rspan, columnspan=cspan, sticky=sticky)
    if tooltip is not None:
        Pmw.Balloon(parent).bind(check_button, balloonHelp=tooltip)
    if selected: check_button.select()
    if disabled: check_button.config(state='disabled')
    return var, check_button

def call_button(parent, command, text='Button', relief='raised', sticky=N+E+W+S, disabled=False, tooltip=None, image=None,
                row=0, column=0, cspan=1, rspan=1, width=20, height=1):
    if image is None:
        call_button=Button(parent, command=command, text=text, width=width, height=height, relief='raised', bd=2)
    else:
        call_button=Button(parent, command=command, image=image, width=width, height=height, relief='raised', bd=2)
    call_button.grid(row=row, column=column, rowspan=rspan, columnspan=cspan, sticky=sticky)
    if disabled:
        call_button.config(state='disabled')
    if tooltip is not None:
        Pmw.Balloon(parent).bind(call_button, balloonHelp=tooltip)
    return call_button

def slider_button(parent, command, span=(1, 100), orient='H', res=1, tooltip=None,
                  row=0, column=0, cspan=1, rspan=1, width=20, height=1):
    orientation='horizontal' if orient=='H' else 'vertical'
    slider=Scale(parent, command=command, from_=span[0], to=span[1], orient=orientation, resolution=res, width=height, length=width)
    slider.grid(row=row, column=column, rowspan=rspan, columnspan=cspan, sticky=NW)
    if tooltip is not None:
        Pmw.Balloon(parent).bind(slider, balloonHelp=tooltip)
    return slider

def entry_field(parent, text='Field', vcmd=lambda args:True, icmd=lambda args:True,
                readonly=False, sticky=N+E+W+S, tooltip=None,
                row=0, column=0, cspan=1, rspan=1, width=20):
    # d: action type, i: index of char to be inserted/deleted, P value of entry given the change is accepted
    # s: value of entry prior to editing, S: string to be inserted or deleted if allowed
    # v: current validation type, V: triggered validation type, w: widget name
    def format_cmd(cmd):
        args='d:%d,i:%i,P:%P,s:%s,S:%S,v:%v,V:%V,W:%W'
        return (parent.register(cmd), args)

    entry_field=Entry(parent, width=width, validate='all', validatecommand=format_cmd(vcmd), invalidcommand=format_cmd(icmd))
    entry_field.grid(row=row, column=column, rowspan=rspan, columnspan=cspan, sticky=sticky)
    entry_field.insert(0, text)
    if readonly:
        entry_field.config(state='disabled')
    if tooltip is not None:
        Pmw.Balloon(parent).bind(entry_field, balloonHelp=tooltip)
    return entry_field

def label_field(parent, text, font='Helvetica', fontsize=14, width=20, bd=5, center=True, sticky=N+E+W+S, tooltip=None,
                row=0, column=0, cspan=1, rspan=1):
    if center:
        justify=CENTER
    else:
        justify=LEFT
    label_variable=StringVar()
    label_variable.set(text)
    label=Label(parent, text=text, textvariable=label_variable, font=(font, fontsize), width=width, bd=bd, justify=justify)
    label.grid(row=row, column=column, rowspan=rspan, columnspan=cspan, sticky=sticky)
    if tooltip is not None:
        Pmw.Balloon(parent).bind(label, balloonHelp=tooltip)
    return label_variable

def input_field(parent, text='File', command=lambda f:True, filetypes=[], mode='file',
                default=None, disabled=False, tooltip=None, ext=None, init_dir=None,
                row=0, column=0, fspan=2, bspan=2, rspan=1, height=1, width=None, sticky=E+W):

    if init_dir is None:
        init_dir=os.getcwd()+'/'
    elif os.path.isdir(init_dir):
        init_dir=init_dir
    elif os.path.isfile(default):
        init_dir=os.path.split(default)[0]
    elif os.path.isdir(default):
        init_dir=default
    else: init_dir=os.getcwd()+'/'

    # Function for changing field
    def change_field(button):
        # Get current path
        thisdir=entry.get()
        if os.path.isfile(thisdir) and thisdir!=default:
            init_dir=os.path.split(thisdir)[0]
        elif os.path.isdir(thisdir) and thisdir!=default:
            init_dir=thisdir
        else:
            init_dir=os.getcwd()+'/'

        # Get User input
        if button.field_mode=='file':
            if ext is None:
                field_input=tkFileDialog.askopenfilename(title=text, initialdir=init_dir, filetypes=filetypes)
            else:
                field_input=tkFileDialog.askopenfilename(title=text, initialdir=init_dir, defaultextension=ext, filetypes=filetypes)
            if type(field_input) is not str or not os.path.isfile(field_input):
                field_input=None
        else:
            field_input=tkFileDialog.askdirectory(title=text, initialdir=init_dir)
            if len(field_input) not in range(2):
                if not os.path.exists(field_input):
                    if tkMessageBox.askyesno('Directory', 'The directory: '+field_input+' does not exist, create it?', icon='warning'):
                        try:
                            os.mkdir(field_input)
                        except:
                            os.makedirs(field_input)
                    else: field_input=None
                if field_input is not None and field_input[-1]!='/':
                    field_input+='/'
            else:
                field_input=None

        # Set field
        if field_input is not None:
            entry.config(state='normal')
            entry.delete(0, len(entry.get())+1)
            entry.insert(0, field_input)
            entry.config(state='disabled', disabledforeground='black')
            entry.xview_moveto(1.0)
            command(entry.get())

    # Make field and button
    if width is None:
        entry=Entry(parent)
    else:
        entry=Entry(parent, width=width)
    entry.grid(row=row, column=column, columnspan=fspan, sticky=sticky)
    if default!=None:
        entry.insert(0, default)
    entry.config(state='disabled', takefocus=' ', disabledforeground='grey' if disabled else 'black', justify='right')
    entry.xview_moveto(1.0)
    button=Button(parent, command=lambda:change_field(button), text=text, bd=2, height=height)
    button.field_mode=mode
    button.grid(row=row, column=column+fspan, rowspan=rspan, columnspan=bspan, sticky=sticky)

    if disabled:
        button.config(state='disabled')
    if tooltip is not None:
        Pmw.Balloon(parent).bind(entry, balloonHelp=tooltip)
        Pmw.Balloon(parent).bind(button, balloonHelp=tooltip)

    return entry, button

def click_entry(parent, text='Field', default='', vcmd=lambda args:True, icmd=lambda args:True,
                sticky=N+E+W+S, disabled=False, invisible=False, tooltip=None,
                row=0, column=0, cspan=2, rspan=1, bw=10, ew=5):
    fspan=cspan-1
    bspan=1

    def activate():
        methods.set_field(field, default)
        field.config(bg='white')
        parent.focus()

    button=call_button(parent, command=activate, text=text, row=row, column=column, cspan=bspan, width=bw, sticky=sticky, disabled=disabled, tooltip=tooltip)
    field=entry_field(parent, text=default if not invisible else '', vcmd=vcmd, icmd=icmd, row=row, column=column+fspan, cspan=fspan, width=ew, readonly=False, sticky=sticky, tooltip=tooltip)
    field.config(disabledforeground='black', disabledbackground=button.config()['background'][-1], highlightcolor='yellow')

    if invisible:
        field.config(relief='flat', highlightthickness=0, state='disabled')
        button.config(relief='flat')

    return field, button

def click_menu(parent, commands=(), text=('Menu',), default=0,
               ondrop=None, sticky='all', tooltip=None,
               row=0, column=0, cspan=1, rspan=1, width=20, height=1):
    if sticky=='all': sticky=N+E+W+S

    def pass_command(c, v):
        menu_variable.set(v)
        c(v)

    # Make Menu Button
    menu_variable=StringVar()
    menu_variable.set(text[default])
    menu_frameing=Frame(parent)
    menu_dropdown=OptionMenu(menu_frameing, menu_variable, *text)
    menu_dropdown['width']=width
    menu_dropdown['height']=height

    if ondrop is not None:
            menu_dropdown.bind('<Button-1>', ondrop)

    # Add commands if any
    if commands!=():
        menu_dropdown['menu'].delete(0, len(text))
        for i, c in enumerate(commands):
            menu_dropdown['menu'].add_command(
                    label=text[i],
                    command=lambda v=text[i]:pass_command(c, v))

    # Define Dimensions
    menu_frameing.grid(row=row, column=column, columnspan=cspan, sticky=sticky)
    menu_dropdown.grid(row=row, column=column, rowspan=rspan, columnspan=cspan, sticky=sticky)
    if tooltip is not None:
        Pmw.Balloon(parent).bind(menu_dropdown, balloonHelp=tooltip)
    return menu_variable, menu_dropdown

def check_menu(parent, title, command=lambda x:True, text=('Menu',), default=0,
               sticky='all', valids=None, colors=None, ondrop=None, tooltip=None,
               row=0, column=0, cspan=1, rspan=1, width=20, height=1):  # Dropdown Checkbox Button
    if sticky=='all': sticky=N+E+W+S

    def pass_check(x):
        check=[c.get() for c in check_variables]
        index=check[:x].count(True)
        color=colors[index] if colors is not None and index in range(len(colors)) else 'black'
        if valids is None:
            valid=True
        elif type(valids) is int:
            valid=check.count(True)<=valids
        elif len(valids)>0:
            valid=check.count(True)<2 or ''.join([str(i) for i in check]) in valids
        else:
            valid=False

        if not valid:
            print 'Error: This selection combination is invalid'
            check_drop_down["menu"].invoke(x)
        else:
            check_drop_down['menu'].entryconfig(x, selectcolor=color)
            command([(t, check[i]) for i, t in enumerate(text)])

    # Make Menu Button
    check_variables=[BooleanVar() for i in text]
    check_menutitle=StringVar()
    check_menutitle.set(title)
    check_menuframe=Frame(parent)

    check_drop_down=OptionMenu(check_menuframe, check_menutitle, *(title,))
    check_drop_down['width']=width
    check_drop_down['height']=height

    if ondrop is not None:
        check_drop_down.bind('<Button-1>', ondrop)

    # Add checkbuttons if any
    check_drop_down['menu'].delete(0)
    for i, t in enumerate(text):
        color=colors[i] if colors is not None and i in range(len(colors)) else 'black'
        check_drop_down['menu'].add_checkbutton(
                label=t, onvalue=True, offvalue=False, selectcolor=color,
                variable=check_variables[i],
                command=lambda x=i:pass_check(x))

    # Set default
    check_drop_down['menu'].invoke(default)

    # Define Dimensions
    check_menuframe.grid(row=row, column=column, rowspan=rspan, columnspan=cspan, sticky=sticky)
    check_drop_down.grid(row=row, column=column, rowspan=rspan, columnspan=cspan, sticky=sticky)

    if tooltip is not None:
        Pmw.Balloon(parent).bind(check_drop_down, balloonHelp=tooltip)

    return check_variables, check_drop_down

def label_menu(parent, text, description, command=lambda x:True,
               sticky='all', tooltip=None, default=0,
               row=0, column=0, cspan=4, rspan=1, width=20, height=2):  # Dropdown Menu Button with changing label
    fspan=cspan-2
    bspan=2
    def pass_command(t):
        descriptor.set(description[text.index(t)])
        command(t)

    descriptor=label_field(parent, text=description[default],
            row=row, column=column, rspan=rspan, cspan=fspan)

    selector=click_menu(parent,
            commands=(lambda t: pass_command(t) for t in text),
            text=text, default=default, tooltip=tooltip,
            row=row, column=column+fspan, rspan=rspan, cspan=bspan, width=width, height=height, sticky=sticky)

    return selector[0]
