"""
    Created Mar 2014
    This file is part of VisualCNA
    Copyright (2014) Prakash Chandra Rathi and Daniel Mulnaes

    VisualCNA is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    VisualCNA is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
#####################################################################
# -*- coding: utf-8 -*-
#
# Authors: Prakash Chandra Rathi and Daniel Mulnaes
# Heinrich Heine University, Duesseldorf
# Institute for Pharmaceutical and Medicinal Chemistry
# Universitaetsstr. 1 40225 Duesseldorf
# Germany
#####################################################################


# Standard python libraries
import os, sys, ctypes, tkMessageBox, site
from Tkinter import *

# External modules
from Pmw import Dialog, NoteBook
from pymol import cmd, stored
from matplotlib import rc

# VisualCNA Modules
modules_path=os.path.dirname(os.path.dirname(__file__))+'/visualCNA_modules'
sys.path.insert(0, modules_path)

# We can't change LD_LIBRARY_PATH dynamically, so we need to load shared object individually
# CNA_HOME=os.path.abspath(os.path.dirname(os.path.dirname(__file__))+"/../../../../..")
CNA_HOME=site.getsitepackages()[0].split('ext')[0]
ctypes.cdll.LoadLibrary(CNA_HOME+'/ext/lib/libopenbabel.so')
ctypes.cdll.LoadLibrary(CNA_HOME+'/ext/lib/libinchi.so')
os.environ['BABEL_LIBDIR']=CNA_HOME+'/ext/lib/openbabel/2.3.1'
os.environ['BABEL_DATADIR']=CNA_HOME+'/ext/share/openbabel/2.3.1'
sys.path.insert(0, CNA_HOME+'/src')

import plot_data, read_data, read_pdb, show_network, widgets, methods, cna_settings, errors

import warnings
# warnings.simplefilter('error', UserWarning)

def __init__(self):
    self.menuBar.addmenuitem('Plugin', 'command', 'VisualCNA', label='VisualCNA', command=lambda s=self:VisualCNA(s))

class VisualCNA:
    def __init__(self, master):
        try:
            if stored.visual_cna_runs==True:
                print 'VisualCNA is already running, if you want to start a new instance of VisualCNA you have to close the existing instance.'
                return
        except:
            pass
        stored.visual_cna_runs=True
        cmd.set('dash_gap', 0.40)
        cmd.set('movie_fps', 4)
        cmd.set('transparency', 0.3)
        cmd.set('retain_order', True)
        cmd.set('pdb_retain_ids', True)
        cmd.bg_color('white')
        def _ray(width=None, height=None, renderer=-1, shift=0):
            # We are overwriting ray because otherwise having a selection will break ray-tracing because of plot update detection
            for s in cmd.get_names('selections'):
                cmd.delete(s)
            if width is None:
                cmd.ray()
            elif height is None:
                cmd.ray(width)
            else:
                cmd.ray(width, height, renderer, shift)

        cmd.extend('ray', _ray)
        self.parent=master.root
        self.hull=Dialog(self.parent, title='VisualCNA', buttons=('Quit VisualCNA',), command=self.quit)
        self.hull.withdraw()
        self.panels=NoteBook(self.hull.interior(), hull_width=900, hull_height=666, raisecommand=self.change_tab)
        self.panels.pack(fill='both', expand=1, padx=3, pady=3)
        self.hull.geometry('725x718+868+0')

        # Set init parameters
        self.kill_state_observer=False
        self.pymol_displ=[]
        self.pymol_const=[]
        self.select_resi=[]
        self.pymol_state=1
        self.pymol_resid=1
        self.observ_freq=0.2  # Frequency of prompting PyMOL for changes, in seconds
        self.dataset_limit=7
        self.process_group=[]
        self.process_pid=[]
        self.process_log={}
        self.cna_args={}
        self.cna_exe=CNA_HOME+"/CNA.sh" if os.path.isfile(CNA_HOME+"/CNA.sh") else None
        self.analyze_settings=cna_settings.analyze_setup()
        self.results_settings=cna_settings.results_setup()
        self.tooltips=cna_settings.tooltips()

        # Prompt user for showing of examples
        show_examples=errors.attention(5, '(This will remove molecules already loaded in PyMol)' if len(cmd.get_names())!=0 else '')

        # Set defaults as show-case examples for Global and Local Indices
        self.data_dir=os.getcwd()+'/'
        self.pdb_file=self.results_settings.pdb_file  #   if show_examples else self.data_dir
        self.net_file=self.results_settings.net_file
        self.example_dir=self.results_settings.example_dir+'1I6W_snt/'
        self.hb_cut_offs=methods.get_hb_cut_offs(self.results_settings.hb_cut_defaults)
        self.hp_cut_offs=methods.get_hp_cut_offs(self.results_settings.hp_cut_defaults, self.hb_cut_offs, self.results_settings.tus_default)
        self.cna_type=self.results_settings.cna_default
        self.wt_dir_name='WT'
        self.mutant_dir_name_prefix='mutant'

        # Initialize Data structures
        self.rd=read_data.Read_data()
        self.pdb_obj=read_pdb.ReadPDB(self.pdb_file)
        if not hasattr(self.pdb_obj, 'atom_details'):
            print 'Example Files not found, we suspect that there was an error in the installation, exiting...'
            return
        self.display=show_network.ShowNetwork(self)

        # Initialize GUI
        self.panel_names={'analyze_panel':'Setup', 'visual_panel':'Analyze', 'modify_panel':'Modify', 'mutate_panel':'Mutate'}
        self.analyze_panel=self.panels.add(self.panel_names['analyze_panel'])
        self.setup()
        self.visual_panel=self.panels.add(self.panel_names['visual_panel'])
        self.analyze(show_examples)
        self.modify_panel=self.panels.add(self.panel_names['modify_panel'])
        self.modify()
        self.mutate_panel=self.panels.add(self.panel_names['mutate_panel'])
        self.mutate(show_examples)

        self.hull.show()
        if show_examples:
            methods.set_field(self.open_data_dir, self.example_dir)
            methods.detect_data_dir_update(self, self.example_dir)
            methods.display_results(self)
            methods.set_field(self.open_data_dir, self.data_dir)
            self.panels.selectpage(self.panel_names['visual_panel'])

        methods.set_field(self.msa_plot.open_file, self.data_dir, True)
        methods.set_field(self.msa_plot.basename, '')
        methods.set_field(self.msa_plot.chain_id, '')
        methods.detect_data_dir_update(self, self.data_dir)

    def quit(self, arg):
        if tkMessageBox.askokcancel('Quit', 'Do you really want to quit VisualCNA?'):
            stored.visual_cna_runs=False
            self.hull.destroy()
            stored.visual_cna_runs=False

    def change_tab(self, tab):
        hull_position=[int(self.hull.geometry().split('+')[1]), int(self.hull.geometry().split('+')[2])]
        tab_height=750
        tab_sizes={self.panel_names['analyze_panel']:(710, tab_height), self.panel_names['modify_panel']:(550, tab_height), self.panel_names['visual_panel']: (900, tab_height), self.panel_names['mutate_panel']:(930, tab_height)}
        self.hull.geometry('%dx%d+%d+%d'%(tab_sizes[tab][0], tab_sizes[tab][1], hull_position[0], hull_position[1]))
        if tab=='Modify':
            cmd.set('mouse_selection_mode', '0')
        else:
            cmd.set('mouse_selection_mode', '1')
        if tab!='Visualize':
            cmd.mstop()

    def setup(self):
        parent_panel=self.analyze_panel

        # Logo
        row=0; column=0; cspan=5
        widgets.image_frame(parent_panel, self.results_settings.logo_image, row=row, column=column, cspan=cspan, rspan=1, width=205, height=75)

        # Simulation Type Parameters
        row=1; column=0; cspan=7
        self.cna_args['cna_type']=widgets.label_menu(parent_panel, text=self.analyze_settings.cna_types,
                description=self.analyze_settings.cna_descrip, default=self.analyze_settings.cna_default,
                command=lambda m: methods.set_cna_type(self, m), tooltip=self.tooltips.analysis_type,
                row=row+1, column=column, cspan=cspan, rspan=2, width=20)

        # Input Parameters
        row=4; column=0; cspan=7
        self.input_label=widgets.label_field(parent_panel, text='Input Settings:',
                fontsize=18, width=50, row=row, column=column, cspan=5, center=False)
        self.open_pdb, self.open_pdb_b=widgets.input_field(parent_panel, text='PDB File',
                tooltip=self.tooltips.pdb_input_file, init_dir=self.data_dir, default=self.data_dir,
                row=row+1, column=column, fspan=cspan-2, bspan=1, disabled=False, filetypes=[('PDB files', '*pdb'), ('All files', '.*')])
        self.open_pdb.config(disabledforeground='grey')
        self.open_pdb_drop, self.open_pdb_drop_m=widgets.click_menu(parent_panel, text=('File',),
                ondrop=lambda e:methods.detect_structure_update(self, 'open_pdb_drop', 'open_pdb_drop_m', defaults=('File',),
                                command=lambda args:methods.set_pdb_input(self, args, 'open_pdb')),
                commands=(lambda args:methods.set_pdb_input(self, args, 'open_pdb'),),
                row=row+1, column=6, cspan=1, width=6, sticky=NW, tooltip=self.tooltips.pdb_input_drop)
        self.open_net, self.open_net_b=widgets.input_field(parent_panel, text='Network',
                tooltip=self.tooltips.net_input_file, init_dir=self.data_dir, default=self.data_dir,
                row=row+2, column=column, fspan=cspan-2, bspan=1, disabled=True, filetypes=[('Network files', '*network.dat'), ('All files', '.*')])
        self.open_net_drop, self.open_net_drop_m=widgets.click_menu(parent_panel, text=('None',),
                ondrop=lambda e:methods.detect_structure_exists(self, 'open_net_drop', 'open_net_drop_m',
                        defaults=('None', 'File'), command=lambda args:methods.set_net_input(self, args, 'open_net')),
                commands=(lambda args:methods.set_net_input(self, args, 'open_net'),),
                row=row+2, column=6, cspan=1, width=6, sticky=NW, tooltip=self.tooltips.net_input_drop)
        self.cna_args['aic'], self.aic_b=widgets.check_button(parent_panel, text=self.analyze_settings.aic_descrip,
                on='--aic', off='', selected=self.analyze_settings.aic_default, sticky=W, tooltip=self.tooltips.akaike_info,
                row=row+3, column=column, cspan=5)
        self.cna_args['fnc_num'], self.fnc_num_b=widgets.click_entry(parent_panel, text='',
                vcmd=lambda args: methods.validate_type(self, args, type_='integer'),
                icmd=lambda args: methods.validate_range(self, args, range_=self.analyze_settings.fnc_range, default=self.analyze_settings.fnc_default),
                default=str(self.analyze_settings.fnc_default), disabled=True, invisible=True, tooltip=self.tooltips.fnc_networks,
                row=row+3, column=5, ew=10, sticky=W)

        # Constraint Parameters Selection
        row=9; column=0; cspan=7
        self.simulation_label=widgets.label_field(parent_panel, text='Constraint Settings:',
                fontsize=18, width=40, row=row, column=0, cspan=5, center=False)
        self.cna_args['tus_type']=widgets.label_menu(parent_panel, text=self.analyze_settings.tus_types,
                description=self.analyze_settings.tus_descrip, default=self.analyze_settings.tus_default,
                command=lambda args:methods.set_hp(self, args), tooltip=self.tooltips.unfolding_type,
                row=row+1, column=0, cspan=cspan, rspan=2, width=20)
        self.cna_args['tet_type']=widgets.label_menu(parent_panel, text=self.analyze_settings.tet_types,
                description=self.analyze_settings.tet_descrip, default=self.analyze_settings.tet_default,
                command=lambda args:methods.set_hp(self, args), tooltip=self.tooltips.tether_placing,
                row=row+3, column=0, cspan=cspan, rspan=3, width=20)

        # H-Bond and Hydrophobic Tether cut-offs
        row=14; column=0; cspan=7
        self.hb_cut_label=widgets.label_field(parent_panel, text='E-cutoff for H-Bonds:',
                row=row+1, column=0, width=20)
        self.cna_args['hb_step'], self.hb_step_b=widgets.click_entry(parent_panel, text='Stepsize' ,
                vcmd=lambda args: methods.validate_type(self, args, type_='numerical'),
                icmd=lambda args: methods.validate_range(self, args, range_=self.analyze_settings.hb_step_range, default=self.analyze_settings.hb_cut_defaults[2]),
                default=str(self.analyze_settings.hb_cut_defaults[2]), tooltip=self.tooltips.hb_step_size,
                row=row+1, column=1, ew=10, sticky=N+S+W)
        self.cna_args['hb_start'], self.hb_start_b=widgets.click_entry(parent_panel, text='Initial',
                vcmd=lambda args: methods.validate_type(self, args, type_='numerical'),
                icmd=lambda args: methods.validate_range(self, args, range_=self.analyze_settings.hb_start_range, default=self.analyze_settings.hb_cut_defaults[0]),
                default=str(self.analyze_settings.hb_cut_defaults[0]), tooltip=self.tooltips.hb_step_min,
                row=row+1, column=3, ew=10, sticky=N+S+W)
        self.cna_args['hb_stop'], self.hb_stop_b=widgets.click_entry(parent_panel, text='Terminal',
                vcmd=lambda args: methods.validate_type(self, args, type_='numerical'),
                icmd=lambda args: methods.validate_range(self, args, range_=self.analyze_settings.hb_stop_range, default=self.analyze_settings.hb_cut_defaults[1]),
                default=str(self.analyze_settings.hb_cut_defaults[1]), tooltip=self.tooltips.hb_step_max,
                row=row+1, column=5, ew=10, sticky=N+S+W)
        self.hp_cut_offs_label=widgets.label_field(parent_panel, text='Distance cutoff for placing Hydophobic Tethers:',
                row=row+2, column=0, width=20, cspan=3)
        self.cna_args['hp_start'], self.hp_start_b=widgets.click_entry(parent_panel, text='Constant' ,
                vcmd=lambda args: methods.validate_type(self, args, type_='numerical'),
                icmd=lambda args: methods.validate_range(self, args, range_=self.analyze_settings.hp_start_range, default=self.analyze_settings.hp_cut_defaults[0]),
                default=str(self.analyze_settings.hp_cut_defaults[0]), tooltip=self.tooltips.hp_step_min,
                row=row+2, column=3, ew=10, sticky=N+S+W)
        self.cna_args['hp_stop'], self.hp_stop_b=widgets.click_entry(parent_panel, text='',
                vcmd=lambda args: methods.validate_type(self, args, type_='numerical'),
                icmd=lambda args: methods.validate_range(self, args, range_=self.analyze_settings.hp_stop_range, default=self.analyze_settings.hp_cut_defaults[1]),
                default=str(self.analyze_settings.hp_cut_defaults[1]), tooltip=self.tooltips.hp_step_max,
                disabled=True, invisible=True, row=row+2, column=5, ew=10, sticky=N+S+W)

        # Unfolding Nuclei Settings
        row=17; column=0; cspan=7
        self.nuclei_label=widgets.label_field(parent_panel, text='Unfolding Nuclei Settings:',
                fontsize=18, width=40, row=row, column=0, cspan=5, center=False)
        self.unf_label=widgets.label_field(parent_panel, text='Unfolding Nuclei Types:',
                row=row+1, column=column, width=20)
        self.cna_args['nuc_1'], self.nuc1_b=widgets.check_button(parent_panel, text=self.analyze_settings.nuc_descrip[0],
                on='1', off='', selected=self.analyze_settings.nuc_defaults[0], tooltip=self.tooltips.unfolding_1,
                row=row+1, column=column+1, cspan=1)
        self.cna_args['nuc_2'], self.nuc2_b=widgets.check_button(parent_panel, text=self.analyze_settings.nuc_descrip[1],
                on='2', off='', selected=self.analyze_settings.nuc_defaults[1], tooltip=self.tooltips.unfolding_2,
                row=row+1, column=column+2, cspan=1)
        self.cna_args['nuc_3'], self.nuc3_b=widgets.check_button(parent_panel, text=self.analyze_settings.nuc_descrip[2],
                on='3', off='', selected=self.analyze_settings.nuc_defaults[2], tooltip=self.tooltips.unfolding_3,
                row=row+1, column=column+3, cspan=1)
        self.cna_args['nuc_4'], self.nuc4_b=widgets.check_button(parent_panel, text=self.analyze_settings.nuc_descrip[3],
                on='4', off='', selected=self.analyze_settings.nuc_defaults[3], tooltip=self.tooltips.unfolding_4,
                row=row+1, column=column+4, cspan=1)
        self.cna_args['tran_src']=widgets.label_menu(parent_panel, text=self.analyze_settings.trs_types,
                description=self.analyze_settings.trs_descrip, default=self.analyze_settings.trs_default, tooltip=self.tooltips.transitions,
                row=row+5, column=0, cspan=cspan, rspan=2, width=20)

        # Output Settings
        row=24; column=0; cspan=7
        self.output_label=widgets.label_field(parent_panel, text='Output Settings:',
                fontsize=18, width=40, row=row, column=column, cspan=5)
        self.cna_args['out_stbmap'], self.out_stbmap_b=widgets.check_button(parent_panel, text='Write stability maps',
                on='--stbmap', off='', selected=self.analyze_settings.output_defaults[0], tooltip=self.tooltips.set_output1,
                sticky=W, row=row+1, column=column, cspan=3)
        self.cna_args['out_ligand'], self.out_ligand_b=widgets.check_button(parent_panel, text='Write ligand',
                on='--write_ligand_mol2', off='', selected=self.analyze_settings.output_defaults[2], tooltip=self.tooltips.set_output2,
                sticky=W, row=row+1, column=column+3, cspan=4)
        self.cna_args['all_results'], self.all_results_b=widgets.check_button(parent_panel, text='Write indices for each structure',
                on='--all_results', off='', selected=self.analyze_settings.output_defaults[1], tooltip=self.tooltips.set_output3,
                sticky=W, disabled=True, row=row+2, column=column, cspan=3)
        self.cna_args['out_network'], self.out_network_b=widgets.check_button(parent_panel, text='Write FIRST Network for single PDB-structure',
                on='--netout', off='', selected=self.analyze_settings.output_defaults[3], tooltip=self.tooltips.set_output4,
                sticky=W, row=row+2, column=column+3, cspan=4)
        self.cna_args['res_dir'], self.open_out_b=widgets.input_field(parent_panel, text='Output',
                default=self.data_dir, tooltip=self.tooltips.set_out_dir, disabled=False, mode='dir',
                row=row+5, column=column, fspan=cspan-2, bspan=2)
        self.run_cna=widgets.call_button(parent_panel, lambda:methods.run_cna(self, self.run_cna, default_label='RUN CNA'),
                text='RUN CNA', tooltip=self.tooltips.run_cna_now,
                row=row+6, column=0, cspan=cspan, height=2)

    def analyze(self, show_examples=True):
        parent_panel=self.visual_panel
        row=0; column=0;
        # Show Data directory
        self.open_data_dir, self.open_data_dir_b=widgets.input_field(parent_panel, text='Data Directory', default=self.data_dir,
                command=lambda args: methods.detect_data_dir_update(self, args), tooltip=self.tooltips.set_datadir,
                row=row, column=column, fspan=7, bspan=3, width=20, disabled=False, mode='dir')
        self.open_data_dir.focus_set()

        # Show PDB directory
        self.show_pdb, self.show_pdb_b=widgets.input_field(parent_panel, text='PDB File',
                init_dir=self.data_dir, filetypes=[('PDB files', '*.pdb'), ('All files', '.*')],
                default=self.pdb_file, tooltip=self.tooltips.show_pdb, disabled=False,
                row=row+1, column=0, fspan=2, bspan=1, width=40)
        self.show_pdb_drop, self.show_pdb_drop_m=widgets.click_menu(parent_panel, text=('File',),
                tooltip=self.tooltips.show_pdb_drop,
                ondrop=lambda e:methods.detect_structure_update(self, 'show_pdb_drop', 'show_pdb_drop_m', defaults=('File',),
                        command=lambda args:methods.set_pdb_input(self, args, 'show_pdb', False)),
                commands=(lambda args:methods.set_pdb_input(self, args, 'show_pdb', False),),
                row=row+1, column=3, cspan=2, width=6, sticky=NW)

        # Show Network directory
        self.show_net, self.show_net_b=widgets.input_field(parent_panel, text='Network',
                init_dir=self.data_dir, filetypes=[('Network files', '*network.dat'), ('All files', '.*')],
                default=self.net_file, tooltip=self.tooltips.show_net, disabled=True,
                row=row+1, column=5, fspan=2, bspan=1, width=40)
        self.show_net_drop, self.show_net_drop_m=widgets.click_menu(parent_panel, text=('PDB',),
                tooltip=self.tooltips.show_net_drop,
                ondrop=lambda e:methods.detect_structure_exists(self, 'show_net_drop', 'show_net_drop_m', defaults=('PDB', 'File'),
                        command=lambda args:methods.set_net_input(self, args, 'show_net')),
                commands=(lambda args:methods.set_net_input(self, args, 'show_net'),),
                row=row+1, column=8, cspan=2, width=6, sticky=NW)

        # Height and width of graphs and buttons
        width=430; height=237

        # Make all graph plots
        self.gi_plot=plot_data.plot_loader(self, parent_panel, 'gi', row=2, column=0, cspan=6, width=width, height=height)
        self.li_plot=plot_data.plot_loader(self, parent_panel, 'li', row=4, column=0, cspan=6, width=width, height=height)
        self.stb_plot=plot_data.stb_loader(self, parent_panel, row=2, column=5, cspan=6, width=width, height=height)
        self.unf_plot=plot_data.unf_loader(self, parent_panel, row=4, column=5, cspan=6, width=width, height=height)

        # Make state slider and draw network button
        row=6; column=0; cspan=5
        self.state_slider=widgets.slider_button(parent_panel,
                command=lambda x: cmd.frame(self.state_slider.get()),
                span=(1, len(self.hb_cut_offs)), tooltip=self.tooltips.set_state,
                row=row, column=column, cspan=cspan, width=435, height=15)

        self.add_more_data, self.add_more_data_b=widgets.check_button(parent_panel,
                on='append', off='overwrite', selected=False, disabled=False,
                text='Add to plots', tooltip=self.tooltips.add_datas,
                row=row, column=column+cspan, rspan=2, cspan=1)
        self.draw_network=widgets.call_button(parent_panel, lambda:methods.display_results(self, mode=self.add_more_data.get()),
                text='Vizualize Data', tooltip=self.tooltips.visualize,
                row=row, column=column+cspan+1, rspan=2, cspan=cspan-1)

        self.open_data_dir.focus_set()

    def modify(self):
        parent_panel=self.modify_panel
        constraint_types=['HB - Hydrogen Bond', 'SB - Salt Bridge', 'HP - Hydrophobic Tether', 'CV - Covaelent Bond', 'SR - Stacked Ring Interaction']
        constraint_barno=['1 Bar', '2 Bars', '3 Bars', '4 Bars', '5 Bars', '6 Bars']

        # Edit Constraint Network in PyMOL
        row=0; column=0; rspan=1
        widgets.label_field(parent_panel, text='Modify Network in PyMOL', fontsize=18,
                row=row, column=column, rspan=rspan, cspan=10, width=40)
        widgets.label_field(parent_panel, text='Edit constraints between selected atom pair:', fontsize=14,
                row=row+1, column=column, rspan=rspan, cspan=10, width=40)
        self.make_constraint_b=widgets.call_button(parent_panel, text='Make Constraint',
                command=lambda:methods.make_pymol_constraint(self), tooltip=self.tooltips.pymol_constraints,
                row=row+2, column=column, rspan=rspan, cspan=2)
        self.make_constraint, self.make_constraint_m=widgets.click_menu(parent_panel,
                text=constraint_types, default=3, tooltip=self.tooltips.select_pymol_type,
                commands=[lambda args:methods.detect_make_constraint_update(self, args)]*len(constraint_types),
                row=row+2, column=column+2, rspan=rspan, cspan=3, width=20)
        self.bar_select, self.bar_select_menu=widgets.click_menu(parent_panel,
                text=constraint_barno, default=4, tooltip=self.tooltips.select_pymol_bars,
                row=row+2, column=column+5, rspan=rspan, cspan=4, width=20)
        self.delete_constraint_b=widgets.call_button(parent_panel,
                text='Delete Constraint', tooltip=self.tooltips.pymol_deletion,
                command=lambda:methods.delete_pymol_constraint(self),
                row=row+3, column=column, rspan=rspan, cspan=2)
        self.undo_button=widgets.call_button(parent_panel,
                text='Undo Change', tooltip=self.tooltips.undo_network_mods,
                command=lambda:methods.constraint_undo(self),
                row=row+3, column=column+2, rspan=rspan, cspan=3)
        self.redo_button=widgets.call_button(parent_panel,
                text='Redo Change', tooltip=self.tooltips.redo_network_mods,
                command=lambda:methods.constraint_redo(self),
                row=row+3, column=column+5, rspan=rspan, cspan=4)

        # Constraint Table
        row=4; column=0; rspan=1
        widgets.label_field(parent_panel, text='Modify Network from Table:', fontsize=18,
                row=row, column=column, rspan=rspan, cspan=10, width=40)
        widgets.label_field(parent_panel, text='Edit constraints from table:', fontsize=14,
                row=row+1, column=column, rspan=rspan, cspan=2, width=20)
        self.table_constraints, self.table_constraints_m=widgets.click_menu(parent_panel,
                text=[i+'s' for i in constraint_types], default=0, tooltip=self.tooltips.select_table_type,
                commands=[lambda args:methods.detect_table_update(self)]*len(constraint_types),
                row=row+1, column=column+2, rspan=rspan, cspan=3, width=20)

        # Search option for Table
        self.prev_search=widgets.call_button(parent_panel,
                text='<-', tooltip=self.tooltips.search_query_last,
                command=lambda: methods.search(self, 'backward'),
                row=row+1, column=column+5, rspan=1, cspan=1, width=1, sticky=W)
        self.table_search=widgets.entry_field(parent_panel, text='', tooltip=self.tooltips.search_query_word,
                row=row+1, column=column+6, rspan=1, cspan=2, width=15, sticky=N+S+W)
        self.next_search=widgets.call_button(parent_panel,
                text='->', tooltip=self.tooltips.search_query_next,
                command=lambda: methods.search(self, 'forward'),
                row=row+1, column=column+8, rspan=1, cspan=2, width=1, sticky=W)
        self.table_frame=widgets.canvas_frame(parent_panel,
                row=row+2, column=0, rspan=rspan, cspan=10, width=460, height=265)

        self.table_canvas=methods.load_table(self, self.table_frame)

        # Add constraint to table
        row=7; column=0; rspan=1
        self.add_to_table=widgets.call_button(parent_panel, text='Add',
                command=lambda:methods.make_table_constraint(self),
                tooltip=self.tooltips.table_constraints,
                row=row, column=column, rspan=rspan, cspan=1, width=2, sticky=W)
        self.add_clabel=widgets.entry_field(parent_panel, text='', readonly=True,
                row=row, column=column+1, rspan=rspan, cspan=2, width=31, sticky=N+S+W)
        self.add_atom_1=widgets.entry_field(parent_panel, text='', tooltip=self.tooltips.set_table_atoms_1,
                vcmd=lambda args: methods.validate_constraint(self, args, range_=self.pdb_obj.atom_id_range, default=''),
                row=row, column=column+3, rspan=rspan, cspan=1, width=8, sticky=N+S+W)
        self.add_atom_2=widgets.entry_field(parent_panel, text='', tooltip=self.tooltips.set_table_atoms_2,
                vcmd=lambda args: methods.validate_constraint(self, args, range_=self.pdb_obj.atom_id_range, default=''),
                row=row, column=column+4, rspan=rspan, cspan=2, width=8, sticky=N+S+W)
        self.add_c_bars , self.add_c_bars_menu=widgets.click_menu(parent_panel,
                text=[str(i) for i in range(1, 7)], default=4, tooltip=self.tooltips.select_table_bars,
                row=row, column=column+6, rspan=rspan, cspan=2, width=3)
        self.add_energy=widgets.entry_field(parent_panel, text='',
                row=row, column=column+7, rspan=rspan, cspan=2, width=8, sticky=N+S+W, readonly=True)

        row=9; column=0; rspan=1
        self.get_pymol_network=widgets.call_button(parent_panel, text='Save Network',
                command=lambda:methods.write_pymol_net(self), tooltip=self.tooltips.save_network_file,
                row=row, column=column, rspan=rspan, cspan=10, height=2)

    def mutate(self, show_examples):
        # Set examples
        msa_example=self.results_settings.msa_file if show_examples else self.data_dir
        name_example=self.results_settings.msa_name if show_examples else ''
        chain_example=self.results_settings.msa_chain if show_examples else ''

        parent_panel=self.mutate_panel
        # Mutate Residues in PyMOL
        row=0; column=0; rspan=1
        widgets.label_field(parent_panel, text='Mutate Residues in PyMOL', fontsize=18,
                row=row, column=column, rspan=rspan, cspan=10, width=40)
        widgets.label_field(parent_panel, text='Mutate residues based on multiple sequence alignment', fontsize=14,
                row=row+1, column=column, rspan=rspan, cspan=10, width=40)

        self.msa_plot=plot_data.msa_loader(self, parent_panel, example_file=msa_example, example_seq_name=name_example, example_chain_id=chain_example,
                row=3, column=0, cspan=10, width=900, height=250)

        row=5; column=0;
        widgets.label_field(parent_panel, text='Compare two data sets (e.g. wt. & mutant)', fontsize=14,
                row=row, column=column, rspan=rspan, cspan=10, width=40)
        self.open_reference_dir, _=widgets.input_field(parent_panel, text='Reference Data Dir', default=self.data_dir,
                tooltip=self.tooltips.open_reference_dir,
                command=lambda args: methods.detect_data_dir_update(self, args),
                row=row+1, column=column, fspan=7, bspan=3, width=20, disabled=False, mode='dir')
        self.open_comparison_dir, _=widgets.input_field(parent_panel, text='Comparison Data Dir', default=self.data_dir,
                tooltip=self.tooltips.open_comparison_dir,
                command=lambda args: methods.detect_data_dir_update(self, args),
                row=row+2, column=column, fspan=7, bspan=3, width=20, disabled=False, mode='dir')
        self.compare=widgets.call_button(parent_panel, lambda:methods.compare_data(self), tooltip=self.tooltips.compare_pairs,
                text='Compare Data Using', row=row+3, column=column, rspan=rspan, cspan=7, height=2)
        self.compare_type, self.compare_type_m=widgets.click_menu(parent_panel,
                text=('SNT Analysis', 'FNC Analysis'), default=0, tooltip=self.tooltips.compare_type,
                commands=[lambda m:methods.set_cna_type(self, m)]*2,
                row=row+3, column=column+7, rspan=rspan, cspan=3, width=20, height=2)

if __name__=='__main__':
    pass
